/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider.fs;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;

import com.zfabrik.components.provider.util.AbstractExtComponentRepository;


/**
 * An abstract component repository implementation over a file system style storage. The defining
 * characteristics is that there is no simpler way to determine some last modified information than
 * a scan for last modified time stamps (up to some depths) over the repository and that
 * scanning is acceptably fast.
 * <p>
 */
public class AbstractFileSystemComponentRepository extends AbstractExtComponentRepository<RootBoundFSCRComponent,MultiRootFSComponentRepositoryDB> {
	private static final String DOT_PROPERTIES = ".properties";
	private static final String SLASH_Z_PROPERTIES = "/z.properties";
	private int checkDepth;
	private Map<String,? extends IAbstractFileSystem> roots;

	public AbstractFileSystemComponentRepository(
			String name,
			int prio,
			int checkDepth
	) {
		this(name,prio,checkDepth,true);
	}
	
	/**
	 * Constructor leaving a choice whether the repo completes configuration in 
	 * the constructor (defaults to true with the other constructors). If not chosen,
	 * the {@link #configure(int)} method needs to be called explicitely before going into
	 * service. This may be useful if {@link #getExpectedConfiguration()} is used with
	 * non-default values or other configuration is changed.
	 */
	public AbstractFileSystemComponentRepository(
			String name,
			int prio,
			int checkDepth,
			boolean autoConfigure
	) {
		super(name,MultiRootFSComponentRepositoryDB.class);
		this.checkDepth = checkDepth;
		if (autoConfigure) {
			configure(prio);
		}
	}


	public AbstractFileSystemComponentRepository(
			String name,
			int prio
	) {
		this(name, prio, Integer.MAX_VALUE / 2); // use (MAX_INT / 2) to avoid integer range wrapping
	}

	public Map<String,? extends IAbstractFileSystem> getRoots() {
		return roots;
	}

	/**
	 * Set roots within repo. Roots must be configured before first scan.
	 * Roots are pathes within the file system based repo without leading slash and
	 * always relative to the repository root folder.
	 */
	public void setRoots(Map<String,? extends IAbstractFileSystem> roots) {
		this.roots = Collections.unmodifiableMap(new LinkedHashMap<>(roots));
	}

	/**
	 * The FSCR is scanning at startup always completely. Therefore we really don't care so much about the
	 * DB in terms of caching. But the DB will be used by workers nevertheless.
	 **/
	public MultiRootFSComponentRepositoryDB scan(MultiRootFSComponentRepositoryDB current) {
		MultiRootFSComponentRepositoryDB db = new MultiRootFSComponentRepositoryDB();
		try {
			fullScan(getName(),db, this.checkDepth);
		} catch (Exception e) {
			throw new RuntimeException("Repository scan failed: "+getName(), e);
		}
		return db;
	}

	/**
	 * Provide all resources for a component into a given folder
	 */
	public void download(RootBoundFSCRComponent c, File folder) {
		IAbstractFileSystem fs = this.roots.get(c.getRoot());
		try {
			if (fs==null) {
				throw new RuntimeException("No root ("+c.getRoot()+") defined");
			}
			Iterator<AbstractFile> all = fs.iterate(c.getName(),Integer.MAX_VALUE);
			byte[] buffer = new byte[16384];
			int l=0;
			int p=c.getName().length();
			while (all.hasNext()) {
				AbstractFile f = all.next();
				File t = new File(folder,f.getName().substring(p));
				if (f.isFolder()) {
					t.mkdirs();
				} else {
					// download the single file
					InputStream in = fs.getInputStream(f.getName());
					try {
						OutputStream out = new FileOutputStream(t);
						try {
							while ((l=in.read(buffer))>=0) out.write(buffer,0,l);
						} finally {
							out.close();
						}
					} finally {
						in.close();
					}
				}
			}
		} catch (Exception e) {
			throw new RuntimeException("Component download failed ("+c.getName()+"): "+getName(), e);
		}
	}


	//
	// complete rebuild of db from abstract fs to checkdepth deep into components
	//
	private void fullScan(String name, MultiRootFSComponentRepositoryDB db, int checkDepth) throws IOException {
		logger.fine("Starting full scan for all roots "+this.roots);
		for (Map.Entry<String,? extends IAbstractFileSystem> root : this.roots.entrySet()) {

			String rootName = root.getKey();
			IAbstractFileSystem fs = root.getValue();

			Map<String,RootBoundFSCRComponent> fsComponents = new HashMap<>();
			// add all to db
			Iterator<AbstractFile> all = fs.iterate("", 3+checkDepth);
			Map<String,Long> lastModified = new HashMap<String, Long>();
			while (all.hasNext()) {
				AbstractFile f = all.next();
				// name relativ to url. E.g. project/x.properties or project/y/z.properties
				String n = f.getName();
				// depth is number of "/"
				int d = depth(n);
				if (d==2 && n.endsWith(SLASH_Z_PROPERTIES)) {
					// its a folder based component
					String cn = n.substring(0,n.length()-SLASH_Z_PROPERTIES.length());
					RootBoundFSCRComponent c = fsComponents.get(cn);
					if (c==null) {
						c = new RootBoundFSCRComponent(cn);
					}
					// even if we had a prop file based component found before, folder wins!
					c.setProperties(readProperties(fs,n));
					c.setHasFolder(true);
					c.setRevision(updateLastModified(lastModified, f, cn));
					c.setRoot(rootName);
					fsComponents.put(c.getName(), c);
				} else
				if (d==1 && n.endsWith(DOT_PROPERTIES)) {
					String cn = n.substring(0,n.length()-DOT_PROPERTIES.length());
					RootBoundFSCRComponent c = fsComponents.get(cn);
					if (c==null) {
						c = new RootBoundFSCRComponent(cn);
						// only if there was no folder based component before
						c.setProperties(readProperties(fs,n));
						c.setRevision(f.getModified());
					}
					c.setHasFile(true);
					c.setRoot(rootName);
					fsComponents.put(c.getName(), c);
				} else
				if ((d==1 && f.isFolder()) || d>=2) {
					// component folder or something inside - we keep updating the revision to come
					// up with the latest in the while scan of the component!
					// determine component name
					int p = n.indexOf('/',n.indexOf('/')+1);
					String cn = (p>=0? n.substring(0,p) : n);
					// fix or memorize last modified
					RootBoundFSCRComponent c = fsComponents.get(cn);
					if (c!=null && c.isHasFolder()) {
						// fix rev
						c.setRevision(updateLastModified(lastModified, f, cn));
					} else {
						updateLastModified(lastModified, f, cn);
					}
				}
			}
			// ok, now we have all for one fs. Let's throw them into the common db
			for (Map.Entry<String,RootBoundFSCRComponent> e : fsComponents.entrySet()) {
				if (!db.getComponents().containsKey(e.getKey())) {
					// only if some earlier root didn't add it already
					// fix the revision to have the root mixed in
					RootBoundFSCRComponent c = e.getValue();
					c.setRevision(longHash(c.getRoot()+c.getRevision()));
					db.putComponent(e.getKey(), c);
				}
			}
		}
		// in conclusion, fix the revisions to include the roots name
		logger.fine("Full scan completed (found "+db.getComponents().size()+" components)");
	}
	
	// just a typical hash https://computinglife.wordpress.com/2008/11/20/why-do-hash-functions-use-prime-numbers/
	private long longHash(String in) {
		long hash = 7;
		int l = in.length();
		for (int i = 0; i < l; i++) {
		    hash = hash*31l + in.charAt(i);
		}
		return hash;
	}

	// update lm map for component, return latest. This is used to get the overall latest
	// modification in a complete component folder
	private long updateLastModified(Map<String, Long> lastModified,AbstractFile f, String cn) {
		Long lm = lastModified.get(cn);
		lm = (lm==null? f.getModified():Math.max(lm,f.getModified()));
		lastModified.put(cn, lm);
		return lm;
	}

	// read props
	private Properties readProperties(IAbstractFileSystem fs, String n) throws IOException {
		String u = n;
		Properties p = new Properties();
		InputStream in = fs.getInputStream(u);
		try {
			p.load(in);
			p.setProperty(COMPONENT_REPO_IMPLEMENTATION, getName());
		} finally {
			in.close();
		}
		return p;
	}


	// compute depth of name (== # of /)
	private static int depth(String n) {
		int a=-1, r = 0;
		while ((a=n.indexOf('/',a+1))>0) r++;
		return r;
	}

	@Override
	public String toString() {
		return super.toString() + ",checkDepth:"+this.checkDepth+",roots:"+roots;
	}

	private final static Logger logger = Logger.getLogger(AbstractFileSystemComponentRepository.class.getName());
}
