/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider.fs;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;


public class FileSystemImpl implements IAbstractFileSystem {
	private File root;
	
	public FileSystemImpl(File root) {
		this.root = root;
	}

	public File getRoot() {
		return this.root;
	}
	
	@Override
	public boolean exists(String name) throws IOException {
		return new File(root,name).exists();
	}

	@Override
	public InputStream getInputStream(String name) throws IOException {
		return new FileInputStream(new File(root,name));
	}

	@Override
	public List<AbstractFile> list(String name, int depth) throws IOException {
		List<AbstractFile> r = new LinkedList<AbstractFile>();
		Iterator<AbstractFile> iaf = iterate(name,depth);
		while (iaf.hasNext()) {
			r.add(iaf.next());
		}
		return r;
	}
	
	@Override
	public Iterator<AbstractFile> iterate(String name, int depth) throws IOException {
		return new ResultIterator(name,new File(root,name),depth);
	}
	
	// in order hierarchical traversal over file hierarchy iterator
	// w/o requiring to read everything in advance
	private class ResultIterator implements Iterator<AbstractFile> {
		private int depth;
		private AbstractFile here;
		private String name;
		private Iterator<AbstractFile> current;
		private Iterator<File> children;
		
		public ResultIterator(String name, File c,int d) {
			this.name=name;
			this.depth=d;
			this.here = new AbstractFile(name,c.isDirectory(),c.lastModified());
			if (c.isDirectory() && d>0) {
				File[] files = c.listFiles();
				this.children = Arrays.asList(files).iterator();
			}
		}
		
		@Override
		public boolean hasNext() {
			if (this.here!=null) {
				return true;
			}
			if (this.current!=null && this.current.hasNext()) { 
				return true;
			}
			if (this.children!=null && children.hasNext()) {
				File c = children.next();
				this.current = new ResultIterator((name.length()>0? name+"/"+c.getName():c.getName()),c,this.depth-1);
				return hasNext();
			}
			return false;
		}
		
		public AbstractFile next() {
			if (hasNext()) {
				if (this.here!=null) {
					try {
						return this.here;
					} finally {
						this.here = null;
					}
				}
				return current.next();
			}
			return null;
		};
		@Override
		public void remove() {
			throw new UnsupportedOperationException();
		}
	}

	@Override
	public String toString() {
		try {
			return "fs-root:"+this.root.getCanonicalPath();
		} catch (IOException e) {
			return "fs-root:"+this.root.getAbsolutePath();
		}
	}
}
