/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.resources;


/**
 * A resource handle is a nameless accessor to a resource managed by the resource management system.
 * <p>
 * The handle can be used to add and remove dependencies of a resource, to adjust the life-cycle 
 * handling of a resource and generally to hold on to a resource.
 * <p>
 * Resources that are not managed
 * by hard references will not be collected as long as an associated resource handle is held onto.
 *  
 *  
 * @author hb
 */

public interface IResourceHandle {
	/**
	 * Reference mode weak, meaning the resource will be observed 
	 * (but not necessarily kept) using a weak reference
	 */
	short WEAK = 0;
	/**
	 * Reference mode soft, meaning the resource will be observed 
	 * (but not necessarily kept) using a soft reference
	 */
	short SOFT = 1;
	/**
	 * Reference mode strong, meaning the resource will be observed 
	 * (and kept) using a strong reference
	 */
	short STRONG = 2; 
	/**
	 * @deprecated user {@link #STRONG}
	 */
	@Deprecated
	short HARD = 2; 
	
	/**
	 * Adjust the life-cycle handling of the single resource
	 *  
	 * @param ttl a time-to-live parameter.
	 * @param exp an absolute point in time. When reaching exp, the resource will be automatically
	 * invalidated. Set to {@link Long#MAX_VALUE} by default.
	 * @param rm the type of reference applied when holding onto the resource (see {@link #WEAK}, {@link #SOFT}, {@link #STRONG}).
	 */
	void adjust(long ttl, long exp, short rm);

	/**
	 * Invalidate the resource.
	 * @param forced if <code>true</code> the resource cannot veto by throwing a ResourceBusyException. Otherwise it can.
	 * @throws ResourceBusyException
	 */
	void invalidate(boolean forced) 
		throws ResourceBusyException;
	
	/**
	 * Retrieve a resource representation adhering to the passed-on type. For components default 
	 * resolutions exist. 
	 * 
	 * @param clz the expected return type
	 */
	<T> T as(Class<T> clz);

	/**
	 * Retrieve a resource representation adhering to the passed-on type reference. For components default 
	 * resolutions exist. The default implementation delegates to {@link #as(Class)} whenever possible.
	 * 
	 * Using the {@link TypeRef} API, it is possible to specify parameterized types in a type-safe manner. For example:
	 * <pre>
	 * <code>
	 * lookup(new TypeRef&lt;List&lt;String&gt;&gt;(){});
	 * </code>
	 * </pre>
	 * 
	 * @param typeRef the expected return type specified via {@link TypeRef} (that also supports parameterized types)
	 */
	<T> T as(TypeRef<T> typeRef);

	/**
	 * Checks whether this handle depends on the resource associated to the passed-in handle.
	 */
	boolean hasDependency(IResourceHandle rh);

	/**
	 * Adds a dependency for this resource. The invalidation or termination of a dependency resource
	 * implies the invalidation of this resource
	 */
	void addDependency(IResourceHandle rh);
	
	/**
	 * Removes a dependency from this resource. 
	 */
	void removeDependency(IResourceHandle rh);
	
	/**
	 * Gets resource info
	 */
	IResourceInfo getResourceInfo();
	
	/**
	 * Get the observer, an id to the handle that does not determine the handles life cycle
	 */
	IResourceObserver getObserver();
	
	/**
	 * Attach the handle to an object. That means, as long as the object is held by some reference, the handle
	 * and accordingly the resource not be collected. This is a good way of propagating the singleton aspect of the handle and
	 * the resource implementation to its emitted objects. 
	 */
	void attach(Object o);

	/**
	 * Detach the handle from an object. See {@link #attach(Object)}. 
	 */
	void detach(Object o);
}
