/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.fs;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * Various file system utilities. 
 */
public class FileUtils {

	private static final int UNZIP_BUFFER = 64 * 1024;
	public static FileFilter ALL_FILES = new FileFilter() {
		public boolean accept(File pathname) {
			return true;
		}
	};

	/**
	 * delete file or folder recursively
	 * 
	 * @param g
	 */
	public static void delete(File g) {
		if (g.isDirectory()) {
			for (File f : g.listFiles()) {
				delete(f);
			}
		}
		g.delete();
	}

	/**
	 * copy files or folders
	 * 
	 * @param from
	 * @param to
	 * @param filter
	 * @throws Exception
	 */
	public static void copy(File from, File to, FileFilter filter) throws Exception {
		_copy(new byte[4096], from, to, (filter == null ? ALL_FILES : filter));
	}

	private static void _copy(byte[] buffer, File from, File to, FileFilter filter) throws Exception {
		if (!from.exists()) {
			throw new IllegalStateException("Copy source does not exist: " + from);
		}
		File t;
		if (from.isDirectory()) {
			to.mkdirs();
			for (File f : from.listFiles(filter)) {
				t = new File(to, f.getName());
				if (f.isDirectory()) {
					_copy(buffer, f, t, filter);
				} else {
					_copySingleFile(buffer, f, t);
				}
			}
		} else {
			if (to.exists() && to.isDirectory()) {
				t = new File(to, from.getName());
			} else {
				t = to;
			}
			_copySingleFile(buffer, from, t);
		}
	}

	private static void _copySingleFile(byte[] buffer, File f, File t) throws FileNotFoundException, IOException {
		FileOutputStream fout = new FileOutputStream(t);
		FileInputStream fin = new FileInputStream(f);
		int l = 0;
		while ((l = fin.read(buffer)) >= 0)
			fout.write(buffer, 0, l);
		fin.close();
		fout.close();
		t.setLastModified(f.lastModified());
	}

	/**
	 * zip a folder to a file
	 */
	public static void zip(File src, File target) throws IOException {
		zip(src, null, target, ALL_FILES);
	}

	/**
	 * zip a folder to a file
	 */
	public static void zip(File src, File target, FileFilter filter) throws IOException {
		zip(src, null, target, filter);
	}

	/**
	 * zip a folder to a file
	 */
	public static void zip(File src, String path, File target) throws IOException {
		zip(src, path, target, ALL_FILES);
	}

	/**
	 * zip a file or folder to a file
	 */
	public static void zip(File src, String path, File target, FileFilter filter) throws IOException {
		OutputStream out = new FileOutputStream(target);
		ZipOutputStream zo = new ZipOutputStream(out);
		try {
			_zip(src, (path==null?"":path), zo, filter);
		} finally {
			zo.close();
		}
	}

	private static void _zip(File src, String path, ZipOutputStream zo, FileFilter filter) throws IOException {
		if (src.isDirectory()) {
			_addFolderToZIP(src, path, zo);
			String p;
			for (File f : src.listFiles(filter)) {
				p = (path.length()>0? path+"/":"") + f.getName();
				_zip(f, p, zo, filter);
			}

		} else {
			_addFileToZIP(src, path, zo);
		}

	}

	private static void _addFolderToZIP(File f, String path, ZipOutputStream zo) throws IOException {
		if (path.length()>0) {
			path = path + "/"; // append to make it appear as a folder
			ZipEntry ze = new ZipEntry(path);
			if (f != null) {
				ze.setTime(f.lastModified());
			}
			zo.putNextEntry(ze);
			zo.closeEntry();
		}
	}

	private static void _addFileToZIP(File f, String path, ZipOutputStream zo) throws IOException {
		ZipEntry ze = new ZipEntry(path);
		zo.putNextEntry(ze);
		ze.setTime(f.lastModified());
		int len = (int) f.length();
		byte[] buffer = new byte[len];
		// read that file completely
		InputStream fin = new FileInputStream(f);
		int l, s = 0;
		while ((l = fin.read(buffer, s, len - s)) > 0)
			s += l;
		fin.close();
		zo.write(buffer);
		zo.closeEntry();
	}

	/**
	 * unzip a file to a folder
	 * 
	 * @param src
	 * @param target
	 */
	public static void unzip(File src, File target) throws IOException {
		InputStream in = new FileInputStream(src);
		ZipInputStream zi = new ZipInputStream(in);
		try {
			String fn;
			ZipEntry ze;
			File f;
			OutputStream out;
			int l;
			byte[] buffer = new byte[UNZIP_BUFFER];
			while ((ze = zi.getNextEntry()) != null) {
				fn = ze.getName();
				if (fn.endsWith("/")) {
					// its a folder
					f = new File(target,fn.substring(0,fn.length()-1));
					f.mkdirs();
				} else {
					// it's a regular file
					f = new File(target, fn);
					f.getParentFile().mkdirs();
					out = new FileOutputStream(f);
					try {
						while ((l = zi.read(buffer)) >= 0) {
							out.write(buffer, 0, l);
						}
					} finally {
						out.close();
					}
				}
			}
		} finally {
			zi.close();
		}
	}
	
	/**
	 * Compute a good file system location for a possibly path style name relative to some base folder.
	 * That is: Slashes will be kept in the path name but characters in path segments will be replaced if considered problematic.
	 * Furthermore, the first path segment will be augmented by a hash representation of the initial path name to 
	 * ensure uniqueness. This method is not transitive, i.e. 
	 * <pre>
	 * computeSafePath(base,"a/b")
	 * </pre>
	 * is not equal to 
	 * <pre>
	 * computeSafePath(new File(base,"a"),"b")
	 * </pre>
	 * 
	 * @param base
	 * @param pathName
	 * @return
	 */
	public static File computeSafePath(File base, String pathName) {
		return new File(base,safePath(pathName));
	}

	/**
	 * Translate a component name into a safe to use relative path in the file system. This is
	 * used to store and organize component data relative to some root folder 
	 */
	private static String safePath(String name) {
		StringBuilder sb = new StringBuilder(name.length()<<1);
		escape(sb,name);
		return sb.toString();
	}
	
	private static void escape(StringBuilder sb,String in) {
		int l = in.length(),a=0,s=0;
		for (int i=0;i<l;i++) {
			char c = in.charAt(i);
			if (c=='/') {
				sb.append(in.substring(a, i));
				a = i+1;
				// make sure segments are no longer than 200 chars
				int sl = sb.length()-s;
				if (sl>200) {
					sb.setLength(s+200);
				}
				s=i+1;
				sb.append(c);
			} else
			if (!isSafe(c)) {
				sb.append(in.substring(a, i));
				a = i+1;
				sb.append("_");
			}
		}
		if (a<l) {
			sb.append(in.substring(a));
		}
	}

	private static boolean isSafe(char c) {
		return (c>='0' && c<='9') || (c>='a' && c<='z') || (c>='A' && c<='Z') || (c=='/') || (c=='_') || (c=='-');
	}


}
