/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.html;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.BitSet;


/**
 * Various encoding and decoding utility functions.
 */
public class Escaper {
	// static stuff for escapeToHTML
	static private BitSet toEscape = new BitSet();
	static BitSet dontNeedEncoding;
	// static final int caseDiff = ('a' - 'A');

	static {
		dontNeedEncoding = new BitSet(256);
		int i;
		for (i = 'a'; i <= 'z'; i++)
			dontNeedEncoding.set(i);
		for (i = 'A'; i <= 'Z'; i++)
			dontNeedEncoding.set(i);
		for (i = '0'; i <= '9'; i++)
			dontNeedEncoding.set(i);
		dontNeedEncoding.set(' '); 
		dontNeedEncoding.set('-');
		dontNeedEncoding.set('_');
		dontNeedEncoding.set('.');
		dontNeedEncoding.set('*');

		// set up a BitSet of characters that need to be encoded
		// note: any char > 190 will be escaped by default
		char[] toBeEscaped = { '\"', '&', '<', '>', 0x40, 169, 174, 182, 188, 189, 190 };

		for (i = 0; i < toBeEscaped.length; i++) {
			toEscape.set(toBeEscaped[i]);
		}
	}

	/**
	 * Escape a string to be used as a JSON String literal. 
	 */
	public static String escapeToJSON(String in) {
		return escapeToJSJSON(in, true);
	}
	
	/**
	 * Escape a string to be used as a Javascript String literal. 
	 */
	public static String escapeToJS(String in) {
		return escapeToJSJSON(in, false);
	}

	private static String escapeToJSJSON(String in, boolean JSON) {
		if (in==null) {
			return null;
		}
		int l = in.length();
		StringBuffer res = new StringBuffer(l<<1);
		char c;
		for (int i = 0; i < l; i++) {
			c = in.charAt(i);
			switch (c) {
			case '\\': 
				res.append('\\').append(c);
				break;
			case '\'': 
				if (JSON) {
					// only needed in JS not JSON
					res.append(c);
					break;
				}
			case '"': 
				res.append('\\').append(c);
				break;
			case '\b': // escape backspace
				res.append("\\b");
				break;
			case '\t': // escape tab
				res.append("\\t");
				break;
			case '\f': // escape line feed
				res.append("\\f");
				break;
			case '\n': // escape newline
				res.append("\\n");
				break;
			case '\r': // escape linefeed
				res.append("\\r");
				break;
			default: 
				if (c<32 || c>=127) {
					// better safe than sorry
					res.append("\\u");
					String hex = Integer.toHexString(c);
					int r=4-hex.length();
					while (--r>=0) {
						res.append('0');
					}
					res.append(hex);
				} else {
					res.append(c);
				}
			}
		}
		return res.toString();
	}


	/**
	 * Escape a string to be used as an HTML String literal
	 */
	public static String escapeToHTML(String in) {
		if (in==null) {
			return null;
		}
		int l = in.length();
		StringBuffer res = new StringBuffer(l<<1);

		String s1 = "&#";
		String s2 = ";";

		char c;
		for (int i = 0; i < l; i++) {
			c = in.charAt(i);
			if ((c > 190) || (toEscape.get(c))) {
				res.append(s1).append(Integer.toString(c)).append(s2);
			} else {
				res.append(c);
			}
		}
		return res.toString();
	}

	/**
	 * URL encode/decode
	 *
	 * This implementation helps addressing two things: 
	 * <ul>
	 * <li>As Apache's mod_proxy unfortunately (and almost utterly incomprehensibly) unescapes URLs before
	 * passing them on to the backend, you can use this method to use an alternative character (e.g. "!") 
	 * rather than "%" as escape character.
	 * </li>
	 * <li>Using this method, you can spare catching UnsupportedEncodingExceptions, that are never thrown 
	 * or you'd be in much bigger trouble anyway. This method always uses utf-8 encoding.
	 * </li>
	 * </ul> 
	 */
	public static String urlEncode(String in, char esc) {
		try {
			String out = URLEncoder.encode(in, "UTF-8");
			if (esc!='%') {
				if (out.indexOf(esc)>0) {
					// it has not been encoded. So we encode it now.
					String p = ""+esc;
					byte[] b = p.getBytes("UTF-8");
					StringBuilder u = new StringBuilder(6);
					for (byte e:b) {
					    u.append('%').append(Character.forDigit((e >> 4) & 0xF, 16)).append(Character.forDigit(e & 0xF, 16));
					}
					out = out.replace(p,u.toString());
				}
				out = out.replace('%',esc);
			}
			return out;
		} catch (UnsupportedEncodingException usee) {
			throw new RuntimeException(usee);
		}
	}

	/**
	 * URL encode/decode
	 *
	 * This implementation helps addressing two things: 
	 * <ul>
	 * <li>As Apache's mod_proxy unfortunately (and almost utterly incomprehensibly) unescapes URLs before
	 * passing them on to the backend, you can use this method to use an alternative character (e.g. "!") 
	 * rather than "%" as escape character.
	 * </li>
	 * <li>Using this method, you can spare catching UnsupportedEncodingExceptions, that are never thrown 
	 * or you'd be in much bigger trouble anyway. This method always uses utf-8 encoding.
	 * </li>
	 * </ul> 
	 */
	public static String urlEncode(String in) {
		return urlEncode(in,'%');
	}

	/**
	 * URL encode/decode
	 *
	 * This implementation helps addressing two things: 
	 * <ul>
	 * <li>As Apache's mod_proxy unfortunately (and almost utterly incomprehensibly) unescapes URLs before
	 * passing them on to the backend, you can use this method to use an alternative character (e.g. "!") 
	 * rather than "%" as escape character.
	 * </li>
	 * <li>Using this method, you can spare catching UnsupportedEncodingExceptions, that are never thrown 
	 * or you'd be in much bigger trouble anyway. This method always uses utf-8 encoding.
	 * </li>
	 * </ul> 
	 */
	public static String urlDecode(String in,char esc) {
		if (esc!='%') {
			in = in.replace(esc,'%');
		}
		try {
			return URLDecoder.decode(in, "UTF-8");
		} catch (UnsupportedEncodingException usee) {
			throw new RuntimeException(usee);
		}
	}
	
	/**
	 * URL encode/decode
	 *
	 * This implementation helps addressing two things: 
	 * <ul>
	 * <li>As Apache's mod_proxy unfortunately (and almost utterly incomprehensibly) unescapes URLs before
	 * passing them on to the backend, you can use this method to use an alternative character (e.g. "!") 
	 * rather than "%" as escape character.
	 * </li>
	 * <li>Using this method, you can spare catching UnsupportedEncodingExceptions, that are never thrown 
	 * or you'd be in much bigger trouble anyway. This method always uses utf-8 encoding.
	 * </li>
	 * </ul> 
	 */
	public static String urlDecode(String in) {
		return urlDecode(in,'%');
	}
		
		
}
