/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.runtime;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Properties;

/**
 * The Foundation class holds basic configuration and the most basic information
 * routines applying to the z2 installation and the running process
 *
 * @author Henning Blohm
 */
public class Foundation {
	// core build number.
	private static Long corebuild;

	
	/**
	 * Returns the core build number. This can be used to compare versions of
	 * the runtime core. It is made up of year, month, day, hour, minute.
	 */
	public static synchronized long getCoreBuildVersion() {
		if (corebuild==null) {
			InputStream in = Foundation.class.getClassLoader().getResourceAsStream(Foundation.class.getPackage().getName().replace(".","/")+"/buildinfo.txt");
			if (in==null) {
				// nothing found - assume we are in development and testing.
				corebuild=Long.MAX_VALUE;
			} else {
				try {
					try {
						Properties p = new Properties();
						p.load(in);
						corebuild = Long.parseLong(p.getProperty("buildno"));
					} finally {
						in.close();
					}
				} catch (IOException ioe) {
					throw new IllegalStateException("Failed to read build no",ioe);
				}
			}
		}
		return corebuild;
	}

	/**
	 * The instances instance id. 
	 */
	private static String instanceId;
	// instance locks (held on for the whole life time of the process)
	@SuppressWarnings("unused")
	private static RandomAccessFile instanceLockFile;
	@SuppressWarnings("unused")
	private static FileLock instanceFileLock;

	/**
	 * Initially determine the instance id and hold on to it.
	 */
	private static String determineInstanceId() {
		if (isWorker()) {
			// determine from system property
			String instanceId=System.getProperty(INSTANCE_ID,"").trim();
			if (instanceId.length()==0) {
				throw new IllegalStateException("Missing instance id specification "+INSTANCE_ID+" for worker process");
			}
			return instanceId;
		} else {
			try {
				Path instances = Paths.get(
					(String) Foundation.getProperties().get(Foundation.HOME_LAYOUT_WORK),
					"instance"
				);
				Files.createDirectories(instances);
				for (int i=0; i<100; i++) {
					String instanceId = String.format("%02d", i);
					RandomAccessFile raf = new RandomAccessFile(instances.resolve(instanceId).toFile(),"rw");
					FileLock lock = raf.getChannel().tryLock();
					if (lock!=null) {
						raf.writeBytes(Long.toString(System.currentTimeMillis()));
						instanceFileLock=lock;
						instanceLockFile=raf;
						System.setProperty(Foundation.INSTANCE_ID, instanceId);
						return instanceId;
					}
				}
			} catch (IOException e) {
				throw new IllegalStateException("Failed to acquire instance lock",e);
			}
			throw new IllegalStateException("Failed to acquire instance lock");
		}
	}

	/**
	 * The runtime instance identifier for the stand alone or home process. This does normally not need to
	 * be specified and should not. Any non-worker process will allocate an instance identifier at 
	 * startup. Instance identifiers make sure that resources that cannot safely be shared among
	 * different z2 runtimes (such as e.g. Java runtime resources) can be 
	 * separated. Instance allocation however is transient and lost at termination of the home or stand alone
	 * z2 process.
	 */
	public static synchronized String getInstanceId() {
		if (instanceId==null) {
			instanceId = determineInstanceId();
		}
		return instanceId;
	}
	
	/**
	 * Returns true if and only if the system is running in development mode.
	 * This is equivalent to the test of the system property {@link Foundation#MODE}.
	 * The method returns true only if this property is set to "development" (case-sensitive check!), any other value (including empty or null) yields to false.
	 *
	 * @return true if system is running in dev-mode, false otherwise
	 */
	public static boolean isDevelopmentMode() {
		boolean result = MODE_DEVELOPMENT.equals(getProperties().get(MODE));
		return result;
	}

    /**
     * System property that defines the z2 home folder. If not set defaults to
     * the environment property Z2_HOME if that is not set either, it will default
     * to "../..".
     */
    public static final String HOME 					= "com.zfabrik.home";

    /**
     * A z2 process may be started in development mode, if setting this
     * system property to the value <code>development</code>.
     * <p>
     * In development mode the runtime may support development specific features like
     * test code compilation
     */
    public final static String MODE					= "com.zfabrik.mode";

    /**
     * Value to system propery {@link #MODE} for the development mode
     */
    public final static String MODE_DEVELOPMENT		= "development";

    /**
     * The system can be run in offline mode. If so, component repositories are requested to avoid
     * attempts to connect to remote repositories. Set this system property to <code>true</code> to run in
     * offline mode. Defaults to <code>false</code>
     */
    public final static String OFFLINE = "com.zfabrik.offline";


    /**
     * System property that can be specified for the home
     * process and will be propagated to worker processes to
     * name the config properties file.
     *
     * This config should strictly contains ONLY configuration that
     * applies to the home and all processes started from there - standalone or embedded.
     *
     * Process level configuration should be passed as system
     * properties
     */
	public final static String CONFIG_FILE 	 		= "com.zfabrik.config";

	/**
	 * Default config file.
	 */
	public final static String CONFIG_FILE_DEF 		= "runtime.properties";

	/**
	 * The home layout to start (sys prop). This is kept for compatibility reasons and simply adds to {@link #HOME_START}
	 */
	public final static String HOME_LAYOUT_COMPONENT 	= "com.zfabrik.home.layout";

	/**
	 * The home system states to attain and keep. Can be specified as comma-separated list
	 */
	public final static String HOME_START 				= "com.zfabrik.home.start";

	/**
	 * file system layout: Folder for temporary data. Using code must use a subfolder!
	 */
    public final static String HOME_LAYOUT_WORK     	= "com.zfabrik.process.folders.work";

    /**
	 * file system layout: Folder for repo caches. Depends on installation folder. Is computed
	 * and available via the foundation
	 */
    public final static String HOME_LAYOUT_REPOS     	= "com.zfabrik.process.folders.repos";

    /**
	 * file system layout: Folder for local data. Depends on installation folder. Is computed
	 * and available via the foundation
	 */
    public final static String HOME_LAYOUT_DATA     	= "com.zfabrik.process.folders.data";

    /**
     * pre-configured process concurrency to be set on the default
     * application thread pool (sys prop)
     */
	public final static String HOME_CONCURRENCY 		= "com.zfabrik.home.concurrency";

	/**
	 * Component name of the worker process, passed as system property to a child process
	 */
	public final static String PROCESS_WORKER 	    = "com.zfabrik.process.worker";

	/**
	 * Instance id of the home process when conveyed to a child process by a system property
	 */
	public final static String INSTANCE_ID = "com.zfabrik.instance.id";

	
    /**
     * In order to provide cluster wide distinguishing and grouping of z2 processes they
     * may share or distringuish by a cluster id
     */
    public final static String HOME_CLUSTER			= "com.zfabrik.home.clusterId";

    /**
     * Language level of the built-in Java compiler. Currently supported values are "6" or "7". Is determined from the JDK used by default.
     */
	public static final String LANGUAGE_LEVEL = "com.zfabrik.java.level";

	/**
	 * If set to <code>BASIC</code> (case independent), the all system properties &lt;protocol&gt;.proxyUser and &lt;protocol&gt;.proxyPassword will be considered
	 * for basic authentication to a proxy server.
	 */
	public static final String PROXY_AUTH = "com.zfabrik.proxy.auth";

	/**
	 * Home layout folder for local pre-built repo
	 */
	public static final String HOME_LAYOUT_LOCAL = "com.zfabrik.process.folders.local";

	/**
	 * Home layout folder for main bin artifacts to start z2
	 */
	public static final String HOME_LAYOUT_BIN = "com.zfabrik.process.folders.bin";

    /**
     * Z2 version
     */
    public static final String Z2_VERSION = "com.zfabrik.version";

    public static Properties getProperties()  {
        return System.getProperties();
    }

    public static boolean isWorker() {
    	return System.getProperty(PROCESS_WORKER)!=null;
    }

    /**
     * Convenience check for the boolean system property {@link #OFFLINE}.
     * @return
     */
    public static boolean isOfflineMode() {
    	return Boolean.getBoolean(OFFLINE);
    }
    
    /**
     * Determine the home folder, that is, the place where repositories will be auto-discovered
     */
    public static File getHomeFolder() {
        return new File(getProperties().getProperty(HOME));
    }
    

}
