/*
 * z2-Environment
 *
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 *
 * contact@zfabrik.de
 *
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider.util;

import java.io.IOException;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.util.expression.X;

/**
 * A utility for component repository implementers. This helps implementing the index
 * of components provided and query functionality (see {@link IComponentsRepository#findComponents(X)} as well as
 * a simple query cache.
 *
 * This class is not thread safe.
 */
public class FSComponentExtRepositoryDB<FSCRC extends FSCRDBComponent> implements Serializable {
	public final static int FIND_CACHE_SIZE = 100;

	public  static final long serialVersionUID = 10L;
	// components known by this repo
	private Map<String, FSCRC> components;

	// everything below is transient

	// immutable version to be handed out
	private transient Map<String, FSCRC> immutableComponents;
	// modules owned by this repo
	private transient Map<String,Integer> modules;
	// immutable version to be handed out
	private transient Map<String,Integer> immutableModules;

	@SuppressWarnings("serial")
	private static class FindCache extends LinkedHashMap<X, Collection<String>> {
		protected boolean removeEldestEntry(Map.Entry<X,java.util.Collection<String>> eldest) {
			return size()>FIND_CACHE_SIZE;
		};
	};

	// simple LRU cache
	private transient FindCache findCache;
	private transient int cacheHits;
	private transient int cacheQueries;
	// --------------------


	public FSComponentExtRepositoryDB() {
		this.components=new HashMap<String, FSCRC>();
		_fixTransients();
	}

	// create a copy
	public FSComponentExtRepositoryDB(FSComponentExtRepositoryDB<FSCRC> db) {
		this.components=new HashMap<String, FSCRC>(db.components);
		_fixTransients();
	}

	// finish de-serialization
	private void readObject(java.io.ObjectInputStream in) throws IOException, ClassNotFoundException {
		in.defaultReadObject();
		_fixTransients();
	}


	private void _fixTransients() {
		this.immutableComponents = Collections.unmodifiableMap(this.components);
		this.modules = new HashMap<String,Integer>();
		for (String cn : this.components.keySet()) {
			_incModule(moduleOf(cn));
		}
		this.immutableModules = Collections.unmodifiableMap(this.modules);
		this.findCache = new FindCache();
	}



	/**
	 * Clear the "find" cache.
	 */
	protected void clearCache() {
		this.findCache=null;
		this.cacheHits=0;
		this.cacheQueries=0;
	}

	/**
	 * Stats of "find" cache.
	 */
	public int getCacheHits() {
		return cacheHits;
	}

	/**
	 * Stats of "find" cache.
	 */
	public int getCacheQueries() {
		return cacheQueries;
	}

	/**
	 * Stats of "find" cache.
	 */
	public int getCacheSize() {
		if (this.findCache!=null) {
			return this.findCache.size();
		}
		return 0;
	}

	/**
	 * Decides whether a component should be allowed to be found
	 * by repos down the priority chain. Defaults to not delegate if the component's
	 * module was found here and to delegate otherwise.
	 *
	 * @param componentName
	 */
	protected boolean delegate(String componentName) {
		return !hasComponentModule(componentName);
	}

	/**
	 * Find locally in this DB (see also {@link IComponentsRepository#findComponents(X)}
	 *
	 * @param propertyExpression
	 * @throws IOException
	 */
	public Collection<String> findComponents(X propertyExpression) throws IOException {
		this.cacheQueries++;
		if (findCache!=null) {
			Collection<String> rs = findCache.get(propertyExpression);
			if (rs!=null) {
				this.cacheHits++;
				// cache hit!
				return new HashSet<String>(rs);
			}
		}
		List<String> r = new LinkedList<String>();
		Object o;
		for (FSCRDBComponent c : this.components.values()) {
			// We search with raw properties
			o = propertyExpression.eval(c.getRawProperties());
			if ((o instanceof Boolean) && (((Boolean) o).booleanValue())) {
				r.add(c.getName());
			}
		}
		if (findCache==null) {
			findCache=new FindCache();
		}
		findCache.put(propertyExpression,r);
		return new HashSet<String>(r);
	}

	/**
	 * Gets rev of local component
	 *
	 * @param cn
	 */
	public long getRevision(String cn) {
		FSCRDBComponent c = components.get(cn);
		if (c!=null) {
			return c.getRevision();
		} else {
			return -1;
		}
	}

	public Map<String, FSCRC> getComponents() {
		return this.immutableComponents;
	}

	/**
	 * Adds a component, updates modules set.
	 */
	public void putComponent(String componentName, FSCRC c) {
		this.components.put(componentName,c);
		_incModule(moduleOf(componentName));
	}

	/**
	 * Remove a component
	 */
	public void removeComponent(String componentName) {
		if (this.components.remove(componentName)!=null) {
			_decModule(moduleOf(componentName));
		}
	}

	/**
	 * Remove a complete module
	 */
	public void removeModule(String moduleName) {
		if (hasModule(moduleName)) {
			String pref = moduleName+"/";
			Iterator<Map.Entry<String, FSCRC>> i = this.components.entrySet().iterator();
			while (i.hasNext()) {
				if (i.next().getKey().startsWith(pref)) {
					i.remove();
				}
			}
			this.modules.remove(moduleName);
		}
	}

	/**
	 * Checks whether this DB has the module
	 */
	public boolean hasModule(String moduleName) {
		return this.modules.containsKey(moduleName);
	}

	/**
	 * Checks whether this DB has the named component's module
	 */
	public boolean hasComponentModule(String componentName) {
		return this.modules.containsKey(moduleOf(componentName));
	}

	/**
	 * gets the modules read-only
	 */
	public Set<String> getModules() {
		return this.immutableModules.keySet();
	}

	/**
	 * Gets all components of a given module
	 */
	public Set<String> getComponentsOfModule(String moduleName) {
		Set<String> r = new HashSet<String>();
		String pref = moduleName+"/";
		for (String cn : this.components.keySet()) {
			if (cn.startsWith(pref)) {
				r.add(cn);
			}
		}
		return r;
	}


	/**
	 * Check and fix modules map
	 */
	private void _decModule(String m) {
		Integer i = this.modules.get(m);
		if (i==1) {
			this.modules.remove(m);
		} else {
			this.modules.put(m,i-1);
		}
	}

	private void _incModule(String m) {
		if (this.modules.containsKey(m)) {
			this.modules.put(m,this.modules.get(m)+1);
		} else {
			this.modules.put(m,1);
		}
	}

	/**
	 * Extracts module name from component name (i.e. the "/" separated prefix)
	 */
	public static String moduleOf(String componentName) {
		int p = componentName.lastIndexOf('/');
		if (p>=0) {
			return componentName.substring(0,p);
		}
		return null;
	}


	@Override
	public int hashCode() {
		return this.components.hashCode();
	}


	@Override
	public boolean equals(Object obj) {
		if (this==obj) {
			return true;
		}
		if (obj instanceof FSComponentExtRepositoryDB) {
			FSComponentExtRepositoryDB<?> db = (FSComponentExtRepositoryDB<?>) obj;
			return db.components.equals(this.components);
		}
		return false;
	}

}
