/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider.util;

import static com.zfabrik.util.fs.FileUtils.computeSafePath;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamClass;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;

import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.util.fs.FileUtils;
import com.zfabrik.util.runtime.Foundation;

/**
 * General repo implementation helper class.
 * Can be forced to purge repo cache on init by
 * setting the system prop <code>com.zfabrik.cr.purge</code>
 *
 * @author hb
 *
 */
public class FSComponentRepositoryHelper<DB extends FSComponentExtRepositoryDB<? extends FSCRDBComponent>> {
	public final static long DEF_EVICTION_DELAY = 24*3600*1000;

	private static final String REPO_PURGE = "com.zfabrik.cr.purge";
//	private final static String IMPL_VERSION = "implv";
	private final static String CORE_VERSION = "corev";
	private File bin_file;
	private File rev_file;
	private File apps, revs, dbf;
	private long evictionDelay = DEF_EVICTION_DELAY;
	private Class<DB> clz;

	/**
	 * Compatibility constructor with empty check properties.
	 */
	public FSComponentRepositoryHelper(Class<DB> clz, File root) {
		this(clz,root,null);
	}

	/**
	 * Initialize a repo helper with a specific DB class, a local store folder and a set of check props that are
	 * to be found again to validate the repository storage. If the latter check fails, local repository content
	 * will be evicted.
	 */
	public FSComponentRepositoryHelper(Class<DB> clz, File root, Properties checkProps) {
		if (System.getProperty(REPO_PURGE) != null) {
			if (root.exists()) {
				logger.info("Purging components repo at " + root);
				FileUtils.delete(root);
			}
		}
		this.clz = clz;
		this.apps = new File(root, "pkg");
		this.revs = new File(root, "rev");
		this.dbf = new File(root, "db");
		if (!root.exists())
			root.mkdirs();
		if (!this.apps.exists())
			this.apps.mkdir();
		if (!this.revs.exists())
			this.revs.mkdir();
		if (!this.dbf.exists())
			this.dbf.mkdir();
		this.bin_file = new File(this.dbf, "db.bin");
		this.rev_file = new File(this.dbf, "db.rev");

		// check whether that whole thing is of a good version, if not, purge
		// that damn repository anyway
		try {
			Lock l = this.lockDB();
			try {

				// new repo is there is no config at all
				boolean newr  = l.properties().isEmpty();
				// 
				// compare previous repository configuration with currently expected
				// configuration and if not matching purge the repository
				//
				Properties expected = new Properties();
				if (checkProps!=null) {
					expected.putAll(checkProps);
				}
				// we force core version
				expected.put(CORE_VERSION,Long.toString(Foundation.getCoreBuildVersion()));
				// we forcefully ignore the providing repo component (as in development this would always lead to purging)
				expected.remove(IComponentsRepository.COMPONENT_REPO_IMPLEMENTATION);
				boolean purge = !expected.equals(l.properties());
				
				try {
					if (!newr && purge) {
						// actually purge!
						Map<Object,Object> diff = new HashMap<>(l.properties().size()+1);
						diff.putAll(expected);
						diff.entrySet().removeAll(l.properties().entrySet()); // compare \ l.properties
						
						logger.info("Found updated repo properties or core version ("+diff.size()+" differences). Will purge local repository now!");
						if (diff.size()<10) {
							
							// simple protection against showing passwords in log
							for (Object key : diff.keySet()) {
								if (key!=null && key.toString().contains("pass")) {
									diff.put(key, "*******");
								}
							}
							logger.info("Expected but not found in previous configuration: "+diff);
						}
						// in any case... new or broken. Clean up and start from scratch
						FileUtils.delete(this.bin_file);
						FileUtils.delete(this.apps);
						this.apps.mkdir();
						FileUtils.delete(this.revs);
						this.revs.mkdir();
					}
				} finally {
					if (newr || purge) {
						// note down new expected config
						l.properties().clear();
						l.properties().putAll(expected);
						l.update();
					}
				}
			} finally {
				l.close();
			}
		} catch (IOException ioe) {
			throw new IllegalStateException("Failed to initialize repo helper at " + root, ioe);
		}
	}

//	public FSComponentRepositoryHelper(Class<DB> clz, File root) {
//		this(clz,root,null,null);
//	}

	public void setEvictionDelay(long evictionDelay) {
		this.evictionDelay = evictionDelay;
	}

	public Lock lockDB() throws IOException {
		Lock l = new LockingRevFile(this.rev_file);
		l.open();
		return l;
	}

	public Lock lockComponent(String component) throws IOException {
		File f = computeSafePath(this.revs, component);
		if (!f.getParentFile().exists()) {
			f.getParentFile().mkdir();
		}
		Lock l = new LockingRevFile(f);
		l.open();
		return l;
	}

	public File getComponentFolder(FSCRDBComponent c) {
		File f = new File(computeSafePath(this.apps, c.getName()), Long.toHexString(c.getRevision()));
		if (!f.exists()) {
			f.mkdirs();
		}
		return f;
	}

	public File getComponentRIFolder(String name) {
		File f = computeSafePath(this.apps, name);
		if (!f.getParentFile().exists()) {
			f.getParentFile().mkdir();
		}
		return f;
	}


	// compare component folder revisions
	// inversely: We want highest revision first
	private final static Comparator<File> REVISION_SORTER = new Comparator<File>() {
		public int compare(File o1, File o2) {
			// we compute signum(o2-o1)
			long r1 = o1.lastModified();
			long r2 = o2.lastModified();
			r1 = r2-r1;
			if (r1>0) return 1;
			if (r1<0) return -1;
			return 0;
		}
	};

	public void evictLocalComponent(String name) throws IOException {
		logger.finer("Checking evictions for local file system of component: " + name);
		try {
			File f = this.getComponentRIFolder(name);
			if (f.exists()) {
				// delete all but newest and only if older than eviction delay
				File[] fs = f.listFiles();
				if (fs.length>0) {
					long now = System.currentTimeMillis();
					Arrays.sort(fs, REVISION_SORTER);
					for (int i=1; i<fs.length;i++) {
						File g = fs[i];
						if (now-g.lastModified()>this.evictionDelay) {
							logger.fine("Evicting local file system of version "+g.getName()+" (aged "+(now-g.lastModified())+"ms, eviction delay is "+this.evictionDelay+"ms) of component: " + name);
							FileUtils.delete(fs[i]);
						}
					}
				}
			}
		} catch (Exception e) {
			logger.warning("Error during local component eviction. The component may still be in use or locked by non-collected resources. Will try again. (" + e + ")");
		}
	}

	public DB readDB() throws IOException {
		if (this.bin_file.exists()) {
			ObjectInputStream oin = new ObjectInputStream(new FileInputStream(this.bin_file)) {
				protected Class<?> resolveClass(ObjectStreamClass desc) throws IOException, ClassNotFoundException {
					return Class.forName(desc.getName(), false, clz.getClassLoader());
				}
			};
			try {
				return clz.cast(oin.readObject());
			} catch (ClassNotFoundException cnfe) {
				IOException ioe = new IOException("DB de-serialization failed");
				ioe.initCause(cnfe);
				throw ioe;
			} finally {
				oin.close();
			}
		} else {
			return null;
		}
	}

	public void saveDB(DB db) throws IOException {
		ObjectOutputStream oout = new ObjectOutputStream(new FileOutputStream(this.bin_file));
		try {
			oout.writeObject(db);
		} finally {
			db.clearCache();
			oout.close();
		}
	}



	private Logger logger = Logger.getLogger(FSComponentRepositoryHelper.class.getName());
}
