/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.home.console;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Handler;
import java.util.logging.LogRecord;

import com.zfabrik.util.logging.CompactFormatter;

/**
 * The log buffer is a flexible home log line buffer with an expiration. It can be expanded and must be equiped with
 * an expiration, that may be prolonged at any time.
 * 
 * Clients may set an expiration and a size and retrieve lines in chunks. The buffer works as a FIFO buffer that
 * evicts any old lines once the requested number of lines was exceeded. Once expiration is reached, the buffer 
 * gets cleared.
 * 
 */
public class LogBuffer extends Handler {
	// default we fall back to
	private int  MAX_SIZE_DEFAULT = 500;
	private long linesHandled =0;
	private int  maxSize = MAX_SIZE_DEFAULT;
	private long expiration=0;
	private LinkedList<String> lines = new LinkedList<>();
	CompactFormatter fm = new CompactFormatter();

	@Override
	public synchronized  void publish(LogRecord record) {
		// count lines processed
		linesHandled++;
		// check expiration
		check();
		if (maxSize>0) {
			try {
				lines.add(fm.format(record));
			} catch (Exception e) {
				lines.add("Failed to format log record ("+e.toString()+")");
			}
		}
		// enforce max size
		setMaxSize(maxSize);
	}
	
	@Override
	public void flush() {
	}
	
	@Override
	public void close() throws SecurityException {
	}

	/**
	 * Reset the buffer to defaults
	 */
	public synchronized  void reset() {
		this.expiration=0;
		check();
	}
	
	public synchronized long getExpiration() {
		return expiration;
	}
	
	public synchronized void setExpiration(long expiration) {
		this.expiration = expiration;
	}
		
	public synchronized int getMaxSize() {
		return maxSize;
	}
	
	public synchronized void setMaxSize(int size) {
		this.maxSize = size;
		while (lines.size()>size) {
			lines.removeFirst();
		}
	}
	
	public synchronized long getLinesHandled() {
		return linesHandled;
	}
	
	public synchronized LogBufferInfo getInfo() {
		LogBufferInfo info = new LogBufferInfo();
		info.setExpiration(expiration);
		info.setLinesHandled(linesHandled);
		info.setMaxSize(maxSize);
		info.setSize(lines.size());
		return info;
	}
	
	/**
	 * fetch a log excerpt starting from overall index and count long. May
	 * be served with less lines (if the log does not have as many as requested)
	 * and may be served with an offset start index, given the buffer does not hold the
	 * requested starting index line anymore.
	 * @param start
	 * @param count
	 * @return
	 */
	public synchronized LogExcerpt get(long start, int count) {
		check();
		// boundaries
		if (start>linesHandled) {
			start = linesHandled;
		}
		int begin =  (int) (lines.size()-(linesHandled-start));
		if (begin<0) {
			// buffer too small, we need to skip some real log
			start = linesHandled - lines.size();
			begin = 0;
		}
		// cap
		count = Math.max(Math.min(count, lines.size()-begin),0);
		List<String> sublist = new ArrayList<String>(count);
		// copy
		lines.subList(begin, begin+count).forEach(sublist::add);
		return new LogExcerpt(start, sublist);
	}
	
	/**
	 * Check on expiration
	 */
	private synchronized void check() {
		if (expiration<System.currentTimeMillis()) {
			// reset everything
			expiration=0;
			setMaxSize(MAX_SIZE_DEFAULT);
		}
	}

	
}
