/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.java.JavaComponentUtil;
import com.zfabrik.components.provider.IComponentFactory;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;

/**
 * the implementation of a component factory resource
 * @author hb
 *
 */
public class ComponentFactoryImpl extends Resource {
	/*
	 *  count invalidations. Used to check for invalidations in ComponentResourceWrapper
	 */
	private static volatile long COUNTER = 0;
	private final static String CF_CLZNAME = IComponentDescriptor.COMPONENT_CLZ;
	
	private String name;
	private CF fac;
	private long counter = COUNTER;

	// maintains the association of the resource with the component fac.
	private static class CF implements IComponentFactory {  
		@SuppressWarnings("unused")
		ComponentFactoryImpl res;
		IComponentFactory    wrp;
		
		public Resource createComponentResource(String name) {	return wrp.createComponentResource(name);	}
	}
	//
	
	// component factory used if the profile simply declares a resource
	private static class ResourceCF implements IComponentFactory {
		private Constructor<? extends Resource> constr;
		
		public ResourceCF(Class<? extends Resource> clz) {
			try {
				this.constr = clz.getConstructor(String.class);
			} catch (SecurityException e) {
				throw new IllegalStateException("Failed to retrieve default constructor from Resource extension",e);
			} catch (NoSuchMethodException e) {
				throw new IllegalArgumentException("Extensions of Resource used as component factory shortcut must have single String arg constructor",e);  
			}
		}

		public Resource createComponentResource(String name) {
			try {
				return this.constr.newInstance(name);
			} catch (IllegalArgumentException e) {
				throw new RuntimeException("Resource construction failed: "+name,e);
			} catch (InstantiationException e) {
				throw new RuntimeException("Resource construction failed: "+name,e);
			} catch (IllegalAccessException e) {
				throw new RuntimeException("Resource construction failed: "+name,e);
			} catch (InvocationTargetException e) {
				throw new RuntimeException("Resource construction failed: "+name,e);
			}
		}
	}
	// 
	
	
	public ComponentFactoryImpl(String name) {
		this.name = name;
	}
	
	public <T> T as(Class<T> clz) {
		if (clz.equals(IComponentFactory.class)) {
			synchronized (this) {
				if (this.fac==null) {
					if (logger.isLoggable(Level.FINE)) {
						logger.fine("Loading Component Factory component "+this.name);
					}
					CF cf = new CF();
					// keep the handle!
					cf.res=this;
					
					// we allow implementations of IComponentFactory or just extension of Resource.
					Class<?> iclz = JavaComponentUtil.loadImplementationClassFromJavaComponent(this.name,CF_CLZNAME,handle());
					if (IComponentFactory.class.isAssignableFrom(iclz)) {
						try {
							cf.wrp = iclz.asSubclass(IComponentFactory.class).newInstance();
						} catch (InstantiationException e) {
							throw new IllegalStateException("Failed to instantiate component factory", e);
						} catch (IllegalAccessException e) {
							throw new IllegalStateException("Failed to instantiate component factory", e);
						}
					} else 
					if (Resource.class.isAssignableFrom(iclz)) {
						iclz.asSubclass(Resource.class);
						cf.wrp = new ResourceCF(iclz.asSubclass(Resource.class));
					}
					this.fac = cf;
				}
				return clz.cast(this.fac);
			}
		}
		if (clz.equals(Long.class)) {
			return clz.cast(counter);
		}
		return null;
	}

	public synchronized void invalidate()	throws ResourceBusyException {
		this.counter = (++COUNTER);
		this.fac = null;
	}

	private final static Logger logger = Logger.getLogger(ComponentFactoryImpl.class.getName());
}
