/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.CodeSigner;
import java.security.CodeSource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.java.JavaComponentClassLoader;
import com.zfabrik.components.java.internal.ClassLoaderIntegration;
import com.zfabrik.resources.IResourceHandle;

/**
 * a class loader implementation satisfying the following requirements:
 * <ul>
 * <li>several delegation targets, avoiding double delegations during the
 * traversal</li>
 * <li>fully transitive</li>
 * <li>fully supporting the URL information provided by URLClassloader</li>
 * </ul>
 * 
 * History:
 * <ul>
 * <li>09.09: Filtering .java resources from the resource pool</li>
 * <li>10.09: Better traversal handling, also avoiding cycle loops and double
 * traversal when traversing foreign loaders, e.g. OSGi loaders</li>
 * </ul>
 * 
 * 
 * @author hb
 * 
 */
public class ComponentClassLoader extends JavaComponentClassLoader {
	private final static Logger logger = Logger.getLogger(ComponentClassLoader.class.getName());
		
	// default is we use the jar file cache
	private final static boolean leaveJarFilesOpen = Boolean.parseBoolean(System.getProperty("com.zfabrik.classloading.openJars","true").trim());
	
	static {
		if (!leaveJarFilesOpen) {
			logger.info("Jar File Cache is disabled");
		}
	}
	
	/*
	 * Jar File LRU cache. Note that there are some concurrency constraints 
	 * we have to observe with respect to closing jar files and handing out
	 * jar file references.
	 */
	private final static int JAR_FILE_CACHE_SIZE = 100;
	
	/*
	 * holder
	 */
	private static class JarFileHolder {
		private JarFile jarFile;
		private Manifest manifest;
		private int lease;
		private boolean dump;
		
		public JarFileHolder(JarFile jf, Manifest manifest) {
			this.jarFile = jf;
			this.manifest = manifest;
		}
		/**
		 * Mark as in use
		 */
		public synchronized boolean checkOut() {
			if (this.jarFile==null) {
				// closed!
				return false;
			}
			this.lease++;
			return true;
		}
		/**
		 * Release from use. If marked for closing, close now
		 */
		public synchronized void release() {
			this.lease--;
			if (this.dump) {
				this.dump();
			}
		}
		/**
		 * Mark for closing of close immediately
		 */
		public synchronized void dump() {
			this.dump=true;
			if (this.lease<=0) {
				try {
					this.jarFile.close();
					if (leaveJarFilesOpen && logger.isLoggable(Level.FINER)) {
						logger.finer("Closed jar file "+this.jarFile.getName());
					}
				} catch (Exception e) {
					logger.log(Level.WARNING,"Error closing jar file "+this.jarFile.getName(),e);
				} finally {
					this.jarFile = null;
				}
			}
		}
		
		public JarFile getJarFile() {
			return jarFile;
		}
		public Manifest getManifest() {
			return manifest;
		}
	}

	/**
	 * An LRU Jar File Cache
	 */
	@SuppressWarnings("serial")
	private static class JarFileCache extends LinkedHashMap<File,JarFileHolder> {
		@Override
		protected boolean removeEldestEntry(Map.Entry<File, JarFileHolder> eldest) {
			if (this.size()>=JAR_FILE_CACHE_SIZE) {
				eldest.getValue().dump();
				return true;
			}
			return false;
		}
		
		/**
		 * Get a holder and check it out. 
		 */
		public synchronized JarFileHolder getJarFile(File f) throws IOException {
			if (leaveJarFilesOpen) {
				stats.jarCacheLookup();
				JarFileHolder jfh = this.get(f);
				if (jfh==null || !jfh.checkOut()) {
					JarFile jf = openJarFile(f);
					jfh = new JarFileHolder(jf,jf.getManifest());
					jfh.checkOut();
					if (logger.isLoggable(Level.FINER)) {
						logger.finer("Opened jar file "+jf.getName());
					}
					this.put(f, jfh);
				} else {
					stats.jarCacheHit();
				}
				return jfh;
			} else {
				stats.jarCacheLookup();
				// simply return a new holder that is marked for dumping
				JarFile jf = openJarFile(f);
				JarFileHolder jfh = new JarFileHolder(jf,jf.getManifest());
				jfh.checkOut();
				jfh.dump();
				return jfh;
			}
		}

		public synchronized void evict(File file) {
			if (file!=null) {
				this.remove(file);
			}
		}

	}

	private static JarFileCache jarFileCache = new JarFileCache();

	private static JarFile openJarFile(File f) throws IOException {
		try {
			return new JarFile(f,false,JarFile.OPEN_READ);
		} catch (IOException ioe) {
			// provide some more diagnostic information
			if (!f.exists()) {
				throw new IOException("Jar file "+f+" does not exist", ioe);
			}
			if (!f.canRead()) {
				throw new IOException("Jar file "+f+" not readable", ioe);
			}
			throw ioe;
		}
	}

	
	/**
	 * Statistics
	 */
	private static class Statistics {
		private final static long LOG_DELAY = 10000;
		
		private long lastLog = 0;
		
		private long getURLsCalls;
		private long getURLsTime;
		private long loadClassCalls;
		private long loadClassTime;
		private long findClassCalls;
		private long findClassTime;
		private long getResourceCalls;
		private long getResourceTime;
		private long jarCacheHits;
		private long jarCacheLookups;

		void loadClass(long time) {
			if (isLog()) {
				this._loadClass(time);
				this.log();
			}
		}

		public void jarCacheLookup() {
			if (isLog()) {
				synchronized (this) {
					this.jarCacheLookups++;
				}
			}
		}

		public void jarCacheHit() {
			if (isLog()) {
				synchronized (this) {
					this.jarCacheHits++;
				}
			}
		}

		synchronized void _loadClass(long time) {
			this.loadClassTime+=time;
			this.loadClassCalls++;
		}

		void getURLs(long time) {
			if (isLog()) {
				this._getURLs(time);
				this.log();
			}
		}

		synchronized void _getURLs(long time) {
			this.getURLsTime+=time;
			this.getURLsCalls++;
		}

		void findClass(long time) {
			if (isLog()) {
				this._findClass(time);
				this.log();
			}
		}
		
		synchronized void _findClass(long time) {
			this.findClassTime+=time;
			this.findClassCalls++;
		}

		void getResource(long time) {
			if (isLog()) {
				this._getResource(time);
				this.log();
			}
		}

		synchronized void _getResource(long time) {
			this.getResourceTime+=time;
			this.getResourceCalls++;
		}

		private boolean isLog() {
			return logger.isLoggable(Level.FINER);
		}
		
		private synchronized void log() {
			if (System.currentTimeMillis()-this.lastLog >= LOG_DELAY) {
				logger.finer(
					"{"+
						"loadClassTime: "+this.loadClassTime+", "+
						"loadClassCalls: "+this.loadClassCalls+", "+
						"findClassTime: "+this.findClassTime+", "+
						"findClassCalls: "+this.findClassCalls+", "+
						"getURLsTime: "+this.getURLsTime+", "+
						"getURLsCalls: "+this.getURLsCalls+", "+
						"getResourceTime: "+this.getResourceTime+", "+
						"getResourceCalls: "+this.getResourceCalls+", "+
						"jarCacheLookups: "+this.jarCacheLookups+", "+
						"jarCacheHits: "+this.jarCacheHits+
					"}"
				);
				this.lastLog = System.currentTimeMillis();
			}
		}
	}
	
	private long getNow() {
		return System.currentTimeMillis();
	}
	
	private static Statistics stats = new Statistics();
	
	/*
	 * tracking helper
	 */
	private static class Tracking extends ClassLoaderIntegration {

		static void add(Object o) {
			Set<Object> s = traversal.get();
			if (s==null) {
				s = new HashSet<Object>();
				traversal.set(s);
			}
			s.add(o);
		}
		
		static boolean isInitial() {
			return traversal.get()==null;
		}
		
		static boolean beenHere(ComponentClassLoader ccl) {
			return (traversal.get()!=null && traversal.get().contains(ccl));
		}
		
		static void endTraversal() {
			traversal.set(null);
		}
	} 

	private static String normalize(String pn) {
		while (pn.startsWith("/")) {
			pn = pn.substring(1);
		}
		while (pn.endsWith("/")) {
			pn = pn.substring(0,pn.length()-1);
		}
		return pn;
	}


	/*
	 * Internally, all resources are identified by a normalized, "/" separated
	 * name without leading or trailing "/"
	 */

	// optimized class path. Reading JAR directories before hand for easier
	// load determination.
	private static class Location {
		public File file; // to dir or jar
		public String entryName; // in dir or jar
		public URL url; // to dir or jar
		
		public Location(File file, String entryName, URL u) throws IOException {
			super();
			this.file = file;
			this.url = u;
			this.entryName = entryName;
		}

		@Override
		public String toString() {
			return "Location [file=" + file + ", entryName=" + entryName + ", url=" + url + "]";
		}
	}

	private class ClassPath {
		private URL[] urls;
		private Map<String, ArrayList<Location>> nameToLocation = new HashMap<String, ArrayList<Location>>();
		private Set<String> packages = new HashSet<>();

		public ClassPath(URL[] urls) {
			this.urls = urls;
			init();
		}

		// read a toc as locations
		private void init() {
			long start=System.currentTimeMillis();
			File f;
			JarFile jf;
			JarEntry je;
			Enumeration<JarEntry> ji;
			String pn;
			for (URL u : urls) {
				try {
					if (!"file".equalsIgnoreCase(u.getProtocol())) {
						throw new IllegalArgumentException(
								"this class loader only supports file URLs");
					}
					f = URLtoFile(u);
					if (f.getName().toLowerCase().endsWith(".jar")) {
						jf = new JarFile(f,false,JarFile.OPEN_READ);
						try {
							ji = jf.entries();
							while (ji.hasMoreElements()) {
								je = ji.nextElement();
								pn = je.getName();
								if (!pn.endsWith(".java")) {
									// HB 09/09: we will note all resources except java
									// files (there should never be a reason to look for
									// them via the loader)
									add(new Location(f, normalize(pn), u), pn.endsWith("/"));
								}
							}
						} finally {
							jf.close();
						}
					} else if (f.isDirectory()) {
						// deep traverse
						_traverse(f, f, u, null);
					}
				} catch (IOException ioe) {
					IllegalStateException ioe2 = new IllegalStateException("Classpath construction - Failed to introspect "+ u.toExternalForm(),ioe);
					throw ioe2;
				}
			}
			// make immutable
			this.packages = Collections.unmodifiableSet(this.packages);
			logger.log(Level.FINE,()->"Classpath construction took "+(System.currentTimeMillis()-start)+"ms: "+ComponentClassLoader.this.name);
		}

		// clp construction
		private File URLtoFile(URL u) {
			try {
				return new File(u.toURI());
			} catch (URISyntaxException use) {
				return new File(u.getPath());
			}
		}

		// clp construction
		private void _traverse(File base, File deep, URL u, String name)
				throws IOException {
			String n;
			for (File g : deep.listFiles()) {
				n = (name == null ? g.getName() : name + "/" + g.getName());
				if (!n.endsWith(".java")) {
					// HB 09/09: we will note all resources except java
					// files (there should never be a reason to look for
					// them via the loader)
					add(new Location(base, n, u), g.isDirectory());
				}
				if (g.isDirectory()) {
					_traverse(base, g, u, n);
				}
			}
		}

		// clp construction
		private void add(Location l, boolean packge) {
			ArrayList<Location> s = this.nameToLocation.get(l.entryName);
			if (s == null) {
				s = new ArrayList<Location>(1);
				this.nameToLocation.put(l.entryName, s);
			}
			if (packge) {
				this.packages.add(l.entryName);
			}
			s.add(l);
		}

		// locate
		public Collection<Location> locateAll(String resPackageName) {
			return this.nameToLocation.get(resPackageName);
		}

		// locate
		public Location locate(String resPackageName) {
			ArrayList<Location> l = this.nameToLocation.get(resPackageName);
			return l!=null? l.get(0):null;
		}
	}
	// END classpath

	// 
	private static CodeSigner[] SIGNER = new CodeSigner[] {};
	private Map<URL, CodeSource> csmap;
	private ClassPath clp;
	private ComponentClassLoader[] newparents;
	private URLClassLoader[] urlparents;
	private ClassLoader[] anyparents;
	private List<ClassLoader> allparents;
	private URL[] resolvedUrls;
	private URL[] urls;
	private long visits = 0;
	private IResourceHandle resource;
	private String name;
	private boolean valid = true;

	// NOTE: Watch out for accumulation of protection domains. Best instantiate
	// a url classloader
	// from a clean acc.

	public ComponentClassLoader(IResourceHandle resource, String name, URL[] urls, ClassLoader[] parents) throws IOException {
		super(new URL[] {});
		this.resource = resource;
		this.name = name;
		this.csmap = new HashMap<URL, CodeSource>(urls.length);
		this.urls = urls;
		this.clp = new ClassPath(urls);
		// sort all our parents
		int n = 0, u = 0, a = 0;
		ClassLoader l;
		for (int i = 0; i < parents.length; i++) {
			l = parents[i];
			if (l!=null) {
				if (l instanceof ComponentClassLoader) {
					n++;
				} else if (l instanceof URLClassLoader) {
					u++;
				} else {
					a++;
				}
			}
		}
		this.newparents = new ComponentClassLoader[n];
		this.urlparents = new URLClassLoader[u];
		this.anyparents = new ClassLoader[a];
		this.allparents = new ArrayList<>(a+n+u);
		a = 0;
		n = 0;
		u = 0;
		for (int i = 0; i < parents.length; i++) {
			l = parents[i];
			if (l!=null) {
				this.allparents.add(l);
				if (l instanceof ComponentClassLoader) {
					this.newparents[n++] = (ComponentClassLoader) l;
				} else if (l instanceof URLClassLoader) {
					this.urlparents[u++] = (URLClassLoader) l;
				} else {
					this.anyparents[a++] = l;
				}
			}
		}
		// freeze
		this.allparents = Collections.unmodifiableList(this.allparents);
	}

	@Override
	public List<ClassLoader> getParents() {
		return allparents;
	}
	
	//
	// URL resolutions
	//

	@Override
	public synchronized URL[] getURLs() {
		long now = getNow();
		try {
			_checkValid();
			if (this.resolvedUrls == null) {
				Set<URL> l = new HashSet<URL>();
				this._addURLs(l);
				this.resolvedUrls = new URL[l.size()];
				l.toArray(this.resolvedUrls);
			}
			return this.resolvedUrls;
		} finally {
			stats.getURLs(this.getNow()-now);
		}
	}

	private synchronized void _addURLs(Set<URL> us) {
		short t;
		if ((t = _preTraversal()) == -1)
			return;
		try {
			for (ComponentClassLoader n : this.newparents) {
				n._addURLs(us);
			}
			for (URLClassLoader u : this.urlparents) {
				us.addAll(Arrays.asList(u.getURLs()));
			}
			us.addAll(Arrays.asList(this.urls));
		} finally {
			if (t == 1) {
				Tracking.endTraversal();
			}
		}
	}

	//
	// the actual class loading
	// 

	protected synchronized Class<?> loadClass(String name, boolean resolve)
			throws ClassNotFoundException {
		Class<?> c;
		String fname = name.replace('.', '/') + ".class";
		try {
			c = _loadClass(fname, name, resolve);
		} catch (Exception e) {
			throw new ClassNotFoundException(name, e);
		}
		if (c == null) {
			// determine  a reason:
			boolean beenhere = Tracking.beenHere(this);
			boolean locally = this.clp.locate(fname)!=null;
			throw new ClassNotFoundException(name+" {loader: "+this.name+", local: "+locally+", valid: "+this.valid+", beenhere: "+beenhere+"}");
		}
		return c;
	}

	private synchronized Class<?> _loadClass(String fname, String name,
			boolean resolve) throws IOException {
		_checkValid();
		long now = 0;
		short t= _preTraversal();
		if (t == -1) {
			return null;		
		} else
		if (t == 1) {
			now = this.getNow();
		}
		this.visits++;
		Class<?> c = null;
		try {
			c = super.findLoadedClass(name);
			if (c == null) {
				for (ComponentClassLoader n : this.newparents) {
					if ((c = n._loadClass(fname, name, resolve)) != null)
						break;
				}
			}
			if (c == null) {
				for (URLClassLoader u : this.urlparents) {
					try {
						c = u.loadClass(name);
						break;
					} catch (ClassNotFoundException cnfe) {
					}
				}
			}
			if (c == null) {
				for (ClassLoader l : this.anyparents) {
					try {
						c = l.loadClass(name);
						break;
					} catch (ClassNotFoundException cnfe) {
					}
				}
			}
			if (c == null) {
				c = _findClass(fname, name);
			}
			if ((c != null) && (resolve)) {
				resolveClass(c);
			}
		} finally {
			if (t == 1) {
				Tracking.endTraversal();
				stats.loadClass(this.getNow()-now);
			}
		}
		return c;
	}

	private void _checkValid() {
		if (!this.valid) {
			throw new IllegalStateException(
					"Loader is not in operation anymore: " + this.toString());
		}
	}
	
	/**
	 * Extract all packages.
	 */
	@Override
	public Collection<String> getClassPathPackages() {
		return this.clp.packages;
	}
	
	/**
	 * Override the default module find class behavior with our findclass (#2143).
	 */
	@Override
	protected Class<?> findClass(String name) throws ClassNotFoundException {
		try {
			Class<?> clz = _findClass(name.replace('.', '/') + ".class",name);
			if (clz==null) {
				throw new ClassNotFoundException(name);
			}
			return clz;
		} catch (IOException e) {
			throw new ClassNotFoundException(name,e);
		}
	}

	private Class<?> _findClass(String fname, String name) throws IOException {
		// no security here, but here would be the place to run a privileged
		// action
		Location l = this.clp.locate(fname);
		if (l != null) {
			// suspend traversal checking for
			// the possibility of recursive loading
			long now = this.getNow();
			Set<Object> s = Tracking.suspendTraversalCheck();
			try {
				byte[] bc = null;
				int i = name.lastIndexOf('.');
				String pkgname = null;
				Package pkg = null;
				if (i >= 0) {
					pkgname = name.substring(0, i);
					pkg = getDefinedPackage(pkgname);
				}
				if (l.file.isDirectory()) {
					// must be a folder containing .class files.
					File cf = new File(l.file, l.entryName);
					// read the byte array
					bc = new byte[(int) cf.length()];
					_readfromStream(new FileInputStream(cf), bc);
					if ((i >= 0) && (pkg == null)) {
						// the package is not yet defined.
						// read the manifest and define the package
						File mf = new File(l.file, "/META-INF/MANIFEST.MF");
						Manifest man = null;
						if (mf.exists()) {
							FileInputStream fin = new FileInputStream(mf);
							try {
								man = new Manifest(fin);
							} finally {
								fin.close();
							}
						}
						if (man != null) {
							definePackage(pkgname, man, l.url);
						} else {
							definePackage(pkgname, null, null, null, null,
									null, null, null);
						}
					}
				} else {
					// must be a jar file
					JarFileHolder jfh = jarFileCache.getJarFile(l.file);
					try {
						JarEntry je = jfh.getJarFile().getJarEntry(l.entryName);
						bc = new byte[(int) je.getSize()];
						_readfromStream(jfh.jarFile.getInputStream(je), bc);
						if ((i >= 0) && (pkg == null)) {
							// the package is not yet defined
							// read the manifest and define the package
							Manifest man = jfh.getManifest();
							if (man != null) {
								definePackage(pkgname, man, l.url);
							} else {
								definePackage(pkgname, null, null, null, null, null, null, null);
							}
						}
					} finally {
						jfh.release();
					}
				}
				if (bc != null) {
					// ok, we have the byte data and defined the package (if
					// applicable)
					// now define the class with the VM
					return defineClass(name, bc, 0, bc.length,_getCodeSource(l.url));
				}
			} finally {
				Tracking.attachTraversalCheck(s);
				stats.findClass(this.getNow()-now);
			}
		}
		return null;
	}

	// read a byte array from a stream and close the stream when done
	private void _readfromStream(InputStream in, byte[] bc) throws IOException {
		int c = 0, l = 0, t = bc.length;
		try {
			while ((t > 0) && ((l = in.read(bc, c, t)) >= 0)) {
				c += l;
				t -= l;
			}
		} finally {
			in.close();
		}
	}

	private CodeSource _getCodeSource(URL u) {
		CodeSource cs = this.csmap.get(u);
		if (cs == null) {
			cs = new CodeSource(u, SIGNER);
			this.csmap.put(u, cs);
		}
		return cs;
	}

	//
	// resource lookup etc.
	// 
	public synchronized URL getResource(String name) {
		_checkValid();
		URL u = null;
		short t = _preTraversal();
		long now=0;
		if (t == -1) {
			return null;
		} else 
		if (t == 1) {
			now = this.getNow();	
		}
		try {
			for (ComponentClassLoader l : this.newparents) {
				if ((u = l.getResource(name)) != null)
					break;
			}
			if (u == null) {
				for (URLClassLoader l : this.urlparents) {
					if ((u = l.getResource(name)) != null)
						break;
				}
			}
			if (u == null) {
				for (ClassLoader l : this.anyparents) {
					if ((u = l.getResource(name)) != null)
						break;
				}
			}
			if (u == null) {
				u = findResource(name);
			}
			return u;
		} finally {
			if (t == 1) {
				Tracking.endTraversal();
				stats.getResource(this.getNow()-now);
			}
		}
	}

	public synchronized URL findResource(String name) {
		_checkValid();
		Location l = this.clp.locate(normalize(name));
		if (l != null) {
			try {
				if (l.file.isDirectory()) {
					return computeFolderURL(l.file,name);
				} else {
					return new URL(new URL("jar:" + l.file.toURI().toURL() + "!/"), name);
				}
			} catch (Exception e) {
			}
		}
		return null;
	}

	/*
	 * folder URLs always end with "/" in URL. We want
	 * to preserve the name that was asked though. Hence we remove the trailing slash.
	 */
	private URL computeFolderURL(File base, String name)
			throws MalformedURLException {
		File d = new File(base,name);
		if (d.isDirectory() && !name.endsWith("/")) {
			String n = normalize(d.toURI().toString());
			return new URL(n);
		}
		return d.toURI().toURL();
	}

	public synchronized Enumeration<URL> getResources(String name)
			throws IOException {
		_checkValid();
		Set<URL> result = new HashSet<URL>();
		_getResources(result, name);
		return Collections.enumeration(result);
	}

	private synchronized void _getResources(Set<URL> result, String name)
			throws IOException {
		short t = _preTraversal();
		long now = 0;
		if (t == -1) {
			return;
		} else
		if (t == 1) {
			now = this.getNow();
		}
		try {
			for (ComponentClassLoader l : this.newparents) {
				l._getResources(result, name);
			}
			Enumeration<URL> e;
			for (URLClassLoader l : this.urlparents) {
				e = l.getResources(name);
				while (e.hasMoreElements()) {
					result.add(e.nextElement());
				}
			}
			for (ClassLoader l : this.anyparents) {
				e = l.getResources(name);
				while (e.hasMoreElements()) {
					result.add(e.nextElement());
				}
			}
			findResources(name, result);
		} finally {
			if (t == 1) {
				Tracking.endTraversal();
				stats.getResource(this.getNow()-now);
			}
		}
	}

	/*
	 * Check if we are the outer caller and if so start tracking our traversal.
	 * Returns -1 if we have been here before. Returns 1 if we are the the outer
	 * caller and 0 otherwise.
	 */
	private short _preTraversal() {
		if (Tracking.isInitial()) {
			Tracking.add(this);
			return 1;
		}
		if (Tracking.beenHere(this)) {
			return -1;
		}
		Tracking.add(this);
		return 0;
	}

	private void findResources(String name, Set<URL> result) throws IOException {
		Collection<Location> s = this.clp.locateAll(normalize(name));
		if (s != null) {
			for (Location l : s) {
				if (l.file.isDirectory()) {
					result.add(computeFolderURL(l.file, name));
				} else {
					result.add(new URL(new URL("jar:" + l.file.toURI().toURL() + "!/"), name));
				}
			}
		}
	}

	public String toString() {
		return super.toString() + ":" + this.name
				+ (this.valid ? " [ok]" : " [OUT OF SERVICE]");
	}

	public synchronized long getVisits() {
		return this.visits;
	}

	public String getName() {
		return this.name;
	}

	public IResourceHandle getJavaComponentHandle() {
		return this.resource;
	}

	public synchronized void invalidate() {
		JavaComponentImpl.logger.fine("Loader going out of service: "+this);
		// close and release jar files
		if (this.clp!=null) {
			for (List<Location> s : this.clp.nameToLocation.values()) {
				for (Location l : s) {
					jarFileCache.evict(l.file);
				}
			}
		}
		this.clp=null;
		this.valid = false;
	}

	
	@Override
	public synchronized boolean isOperational() {
		return this.valid;
	}
	
}
