/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.resources;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.IResourceObserver;

/*
 *  Additional data, in particular dependency information, associated with a resource.
 *  In essence the ResourceEnv represent the actual resource graph, independently on 
 *  whether resources are held by strong refs or otherwise.
 *  The graph is updated after garbage collections via the tick() cleanup.
 *    
 *  Concurrency control for deps: synchronize from target to source
 *  
 */
public class ResourceEnv implements IResourceObserver {
	
	protected NamespaceImpl  ns;
	protected ResourcesMap.ValueHolder vh;
	
	public ResourcesMap.ValueHolder getValueHolder() {
		return vh;
	}


	private Set<ResourceEnv> dependencies;
	private Set<ResourceEnv> dependants;
	
	public ResourceEnv(NamespaceImpl namespace, ResourcesMap.ValueHolder vh) {
		super();
		this.ns = namespace;
		this.vh = vh;
	}

	public IResourceHandle getHandle() {
		return this.vh.getValue();
	}
	
	public String getName() {
		return this.vh.getKey();
	}
	
	public boolean hasDependency(ResourceEnv target) {
		synchronized (this) {
			return (this.dependencies!=null && this.dependencies.contains(target));
		}
	}
	
	public void addDependency(ResourceHandleImpl target) {
		ResourceEnv tenv = target.env;
		synchronized (tenv) {
			synchronized (this) {
				// put new dependency
				if (tenv.dependants==null) {
					tenv.dependants = new HashSet<ResourceEnv>();
				}
				tenv.dependants.add(this);
				if (this.dependencies==null) {
					this.dependencies = new HashSet<ResourceEnv>();
				}
				this.dependencies.add(target.env);
				ResourceHandleImpl rhi = this.vh.getValue();
				if (rhi!=null) {
					if (rhi.deps==null) {
						rhi.deps = new HashSet<ResourceHandleImpl>();
					}
					rhi.deps.add(target);
				}
			}
		}
	}

	public void removeDependency(ResourceEnv tenv) {
		synchronized (tenv) {
			synchronized (this) {
				boolean removed=false;
				if (this.dependencies!=null) {
					removed = this.dependencies.remove(tenv);
					if (this.dependencies.isEmpty()) {
						this.dependencies=null;
					}
				}
				if (removed) {
					ResourceHandleImpl rhi = this.vh.getValue();
					if (rhi!=null) {
						if (rhi.deps!=null) {
							rhi.deps.remove(tenv.vh.getValue());
							if (rhi.deps.isEmpty()) {
								rhi.deps=null;
							}
						}
					}
					if (tenv.dependants!=null) {
						if (!tenv.dependants.remove(this)) {
							throw new IllegalStateException("Found missing env dependency "+this+" --> "+tenv);
						}
						if (tenv.dependants.isEmpty()) {
							tenv.dependants=null;
						}
					}
				}
			}
		}
	}

	/*
	 * get one dependant and remove it from the dependency edge
	 */
	public synchronized ResourceEnv popDependant() {
		if (this.dependants != null && !this.dependants.isEmpty()) {
			Iterator<ResourceEnv> i = this.dependants.iterator();
			ResourceEnv env = i.next();
			this.dependants.remove(env);
			// remove as dependency
			_removeThisFromDependant(env);
			return env;
		}
		return null;
	}

	/*
	 * get a snapshot of all dependencies
	 */
	public synchronized Set<ResourceEnv> getDependants() {
		if (this.dependants != null) {
			return new HashSet<ResourceEnv>(this.dependants);
		}
		return Collections.emptySet();
	}
	
	/*
 	 * Remove dependencies and dependants from a resource (called after invalidation). 
	 */
	public void removeDeps() {
		Set<ResourceEnv> deps=null;
		synchronized (this) {
			if (this.dependants!=null) {
				for (ResourceEnv env : this.dependants) {
					_removeThisFromDependant(env);
				}
				this.dependants=null;
			}
			if (this.dependencies==null) {
				return;
			}
			deps = new HashSet<ResourceEnv>(this.dependencies);
		}
		// sync is always from dependency to dependant. So we cannot lock here.
		for (ResourceEnv env : deps) {
			this.removeDependency(env);
		}
	}

	/*
	 * remove a dependency on this from a dependant. Not the synchronization rules: always from dependency to dependant!
	 */
	private void _removeThisFromDependant(ResourceEnv env) {
		synchronized (env) {
			if (env.dependencies!=null) {
				if (!env.dependencies.remove(this)) {
					throw new IllegalStateException("Found missing env dependency "+env+" --> "+this);
				}
				if (env.dependencies.isEmpty()) {
					env.dependencies=null;
				}
			} 
		}
	}

	public String toString() {
		return this.vh.getKey()+(this.vh.getValue()==null? " (value collected)":"");
	}

}
