/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.work;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.zfabrik.work.IThreadPool;
import com.zfabrik.work.WorkManager;

public class WorkManagerImpl extends WorkManager {
	private long totThreadCount = 0;

	private WorkManagerImpl() {
	}

	public static void initialize() {
		if (WorkManager.instance == null) {
			WorkManager.instance = new WorkManagerImpl();
			ApplicationThreadPoolImpl.initialize();
		}
	}
	
	public static void shutdown() {
		WorkManagerImpl i = (WorkManagerImpl) WorkManager.instance;
		try {
			if (i!=null) {
				Set<String> tps = new HashSet<String>();
				while (true) {
					tps.clear();
					synchronized (i.threadPools) {
						tps.addAll(i.threadPools.keySet());
						if (tps.isEmpty()) {
							break;
						}
					}
					for (String s : tps) {
						try {
							i.releaseThreadPool(s);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				} 
			}
		} finally {
			WorkManager.instance=null;
		}
	}

	@Override
	public IThreadPool createThreadPool(String id) {
		synchronized (threadPools) {
			if (this.threadPools.keySet().contains(id))
				throw new IllegalArgumentException("thread pool \"" + id + "\" does already exist");
			ThreadPoolImpl tp = new ThreadPoolImpl(id, this);
			this.threadPools.put(id, tp);
			return tp;
		}
	}

	@Override
	public void releaseThreadPool(String id) {
		synchronized (threadPools) {
			ThreadPoolImpl tp = this.threadPools.remove(id);
			if (tp == null)
				throw new IllegalArgumentException("thread pool \"" + id + "\" does not exist");
			tp.shutdown();
		}
	}

	//
	//
	private Map<String, ThreadPoolImpl> threadPools = new HashMap<String, ThreadPoolImpl>();
	private final static int POOLSIZE = 20;
	private WorkerThread[] pool = new WorkerThread[POOLSIZE];
	private int pooltop = 0;

	// called by a thread pool impl when a job has been completed and it is
	// ready
	// to take another one. Returns whether to go on wait
	protected boolean taskCompleted(WorkerThread thread) {
		// first reset context class loader (as not to hold on to releasable resources)
		thread.setContextClassLoader(null);
		// secondly check if there is more tasks for the threadpool this thread was
		// working on...
		ThreadPoolImpl tpi = thread.getThreadpool();
		Runnable task = tpi.fetchTaskOrQuit();
		if (task != null) {
			// yes! associate and continue right away
			thread.setRunnable(task);
			return false;
		} else {
			synchronized (pool) {
				if (pooltop == POOLSIZE) {
					// pool is full. simply let this thread die
					return false;
				} else {
					// return to pool
					pool[pooltop++] = thread;
					return true;
				}
			}
		}
	}

	public void start(final ThreadPoolImpl tpimpl, final Runnable runnable) {
		WorkerThread t;
		synchronized (pool) {
			if (pooltop == 0) {
				// create new
				/*
				 * By executing the initialization in a privileged action, our classloaders
				 * get a fresh accesscontrolcontext. Otherwise, they would hold on to 
				 * potentially several protection domains, which in turn hold on to other
				 * class loaders (traversed on the call stack) that may not be valid anymore (i.e. a leak!)
				 */
				t = AccessController.doPrivileged(new PrivilegedAction<WorkerThread>(){
					public WorkerThread run() {
						return new WorkerThread("worker thread " + (++totThreadCount),WorkManagerImpl.this, tpimpl, runnable);
					}
				});
				t.start();
			} else {
				// try to take from pool
				t = pool[--pooltop];
				pool[pooltop]=null;
				t.setRunnable(runnable);
				t.setThreadpool(tpimpl);
				t.kick();
			}
		}
	}

}
