/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.launch;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.home.console.TableHelper;
import com.zfabrik.home.console.gui.ManagementConsole;
import com.zfabrik.launch.impl.ProcessRunnerImpl;
import com.zfabrik.launch.impl.Z2CommandLine;
import com.zfabrik.sync.SynchronizationRunner;
import com.zfabrik.util.internal.WorkerVault;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.util.sync.ISynchronizer;
import com.zfabrik.util.threading.TimerUtil;
import com.zfabrik.work.WorkUnit;

/**
 * Home Process Launcher of z2.
 *
 * @author Henning Blohm
 */
public class HomeLauncher extends Z2CommandLine {
	/**
	 * The home synchronizer is house keeper of what to keep up to make it a home!
	 */
	private static final String HOME_SYNCHRONIZER = "com.zfabrik.boot.main/homeSynchronizer";

	public static HomeLauncher INSTANCE;

	public final static String HOME_CONCURRENCY 	= "com.zfabrik.home.concurrency";
	public final static String HOME_AUTO_VERIFY 	= "com.zfabrik.home.autoVerifyInterval";

	// initial runtime states, to be reached initially
	private boolean noprompt = false;
	private boolean gui = false;
	private boolean pause = false;
	private boolean stopping = false;
	private int autoVerify = 0;
	private ManagementConsole mc;
	private Timer autoVerifier;
	private AutoVerifyTask autoVerifyTask;

	/*
	 * Automatic recurring verification of the home system states.
	 */
	private class AutoVerifyTask extends TimerTask {
		public void run() {
			try {
				if (mc!=null) {
					mc.verify();
				} else {
					new SynchronizationRunner(SynchronizationRunner.VERIFY_ONLY).execute(true);
				}
			} catch (Exception e) {
				logger.log(Level.WARNING,"Error during scheduled verfication",e);
			}
		}
	}

	/*
	 * shutdown hook: bring the system down when the home process is
	 * supposed to stop - in an orderly fashion.
	 */
	private static class ShutdownHook implements Runnable {
		private HomeLauncher main;

		public ShutdownHook(HomeLauncher main) {
			this.main = main;
		}

		public void run() {
			synchronized (main) {
				if (main.stopping) return;
			}
			System.err.println("Received shutdown signal: Forced termination");
			main.shutdown();
		}
	}

	/*
	 * Constructor
	 */
	public HomeLauncher(String[] args) throws Exception {
		super(args);
		INSTANCE = this;
	}

	public static void main(String[] args) throws Exception {
		new HomeLauncher(args).run();
	}

	private void run() throws Exception {

		this.pause = getParams().keySet().remove("-pause");
		this.noprompt = getParams().keySet().remove("-np");
		this.gui = getParams().keySet().remove("-gui");
		
		if (!getParams().isEmpty()) {
			log.severe("Unknown options: "+getParams().keySet());
			usage();
		}

		if (this.pause) {
			System.out.println("Please press ENTER to continue");
			new BufferedReader(new InputStreamReader(System.in)).readLine();
		}

		WorkUnit.initCurrent();
		// let's assume the worst
		int exitCode = -1;
		try {

			if (this.gui) {
				mc = new ManagementConsole();
				mc.init();
			}

			// instantiate process master and initialize kernel
			ProcessRunnerImpl.start();

			// fix title to display home id
			if (mc!=null) {
				mc.setTitle("Z2 Home ("+Foundation.getProperties().getProperty(Foundation.HOME_CLUSTER,"<unknown>")+")");
			}

			// check automatic verification
			synchronized (this) {
				try {
					String av = Foundation.getProperties().getProperty(HOME_AUTO_VERIFY);
					if (av!=null) {
						this.autoVerify = Integer.parseInt(av);
						this.autoVerifier = TimerUtil.createTimer("Auto Verifier",true);
					}
				} catch (NumberFormatException nfe) {
					throw new IllegalArgumentException("Failed to initialize auto verify", nfe);
				}
			}

			// have the home synchronizer complete
			// which implies attaining home target states
			ISynchronizer s = IComponentsLookup.INSTANCE.lookup(HOME_SYNCHRONIZER, ISynchronizer.class);
			if (s == null) {
				throw new IllegalStateException("Failed to lookup home synchronizer " + HOME_SYNCHRONIZER);
			}
			s.complete(null);

			// close the current workunit
			WorkUnit.closeCurrent();
			logger.info("Completed home process initialization");

			// register shutdown hook
			Thread shutdown = new Thread(new ShutdownHook(this), "shutdown hook");
			Runtime.getRuntime().addShutdownHook(shutdown);

			// schedule automatic verification
			if (this.autoVerify>0) {
				setAutoVerify(true);
			}

			//
			// begin of operational mode
			//

			if (mc!=null) {
				// we wait for the mc to tell us when to go down
				mc.waitFor();
			} else
			if (this.noprompt) {
				synchronized (this) {
					this.wait();
				}
			} else {
				// console interface:
				String l;
				BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
				boolean quit = false;
				do {
					try {
						System.out
								.println("\npress (q)uit, (c)omponents, (f)oundation props, (s)ystem props, s(y)nc repo, (v)erify, (g)c:>");
						l = in.readLine();
						if (l!=null) {
							l=l.toLowerCase();
							if ("q".equals(l)) {
								quit = true;
							} else if ("c".equals(l)) {
								// print loaded apps
								System.out.println("loaded components:\n");
								TableHelper.printResourceList(IComponentsLookup.INSTANCE);
							} else if ("s".equals(l)) {
								TableHelper.printprops(System.getProperties());
							} else if ("f".equals(l)) {
								TableHelper.printprops(Foundation.getProperties());
							} else if ("y".equals(l)) {
								new SynchronizationRunner(SynchronizationRunner.INVALIDATE_AND_VERIFY).execute(true);
							} else if ("v".equals(l)) {
								new SynchronizationRunner(SynchronizationRunner.VERIFY_ONLY).execute(true);
							} else if ("g".equals(l)) {
								System.out.println("Calling System.gc()");
								System.gc();
							} else {
								if (!"".equals(l))
									System.out.println("don't understand: " + l);
							}
						} else {
							// exit
							quit=true;
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				} while (!quit);
			}
			// ok, we are leaving in good shape
			exitCode=0;
		} catch (Exception e) {
			logger.log(Level.SEVERE, "Terminating because of exception", e);
			if (mc!=null) {
				mc.severe(e);
			}
		} catch (Throwable e) {
			logger.log(Level.SEVERE, "Terminating hard because of error", e);
		} finally {
			try {

				// close the current work unit (that is bound to the main
				// thread.
				WorkUnit.closeCurrent();
			} finally {
				//
				// end of operational mode
				//
				shutdown();
				// swing hangs sometimes. So give it a final blow.
				System.exit(exitCode);
			}
		}
	}

	public synchronized boolean hasAutoVerify() {
		return (this.autoVerifier!=null);
	}

	public synchronized void setAutoVerify(boolean on) {
		if (hasAutoVerify()) {
			if (!on) {
				logger.info("Cancelling automatic verification runs.");
			}
			if (this.autoVerifyTask!=null) {
				this.autoVerifyTask.cancel();
				this.autoVerifyTask = null;
			}
			if (on) {
				this.autoVerifyTask = new AutoVerifyTask();
				long d = this.autoVerify*1000;
				logger.info("Scheduling automatic verification runs every "+this.autoVerify+"s.");
				this.autoVerifier.scheduleAtFixedRate(this.autoVerifyTask, d, d);
			}
		}
	}

	private void shutdown() {
		synchronized (this) {
			if (this.stopping)
				return;
			this.stopping = true;
		}
		logger.info("Shutting down home process...");
		// anything that is currently starting will be killed
		WorkerVault.INSTANCE.shutDown();
		// otherwise try a clean shut down
		try {
			try {
				if (this.autoVerifier!=null) {
					this.autoVerifier.cancel();
				}
			} finally {
				ProcessRunnerImpl.stop();
				if (mc!=null) {
					mc.leave();
					mc.dispose();
				}
			}
		} catch (Throwable t) {
			t.printStackTrace();
		}
		synchronized (this) {
			// in case of "no prompt" we wait on this
			this.notifyAll();
		}
	}

	//
	// logging
	//
	public static final Logger logger = Logger.getLogger(HomeLauncher.class.getName());
}
