/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.launch.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.nio.charset.StandardCharsets;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.logging.Handler;
import java.util.logging.Logger;

import javax.security.auth.login.Configuration;

import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.home.console.HomeMXBeanImpl;
import com.zfabrik.impl.components.ComponentsManager;
import com.zfabrik.impl.resources.ResourceManagerImpl;
import com.zfabrik.impl.work.ApplicationThreadPoolImpl;
import com.zfabrik.impl.work.WorkManagerImpl;
import com.zfabrik.util.function.io.IO;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.util.threading.ThreadUtil;
import com.zfabrik.work.WorkUnit;

/**
 * Core process starter. A process that wants to be a z2 process simply
 * starts and stops its z2 nature using the class.
 * <p>
 * As an API user, use the class <code>com.zfabrik.launch.ProcessRunner</code> instead.
 * 
 */
public class ProcessRunnerImpl {
	
	private static final String VERSION_FILE = "VERSION.txt";


	/**
	 * The essential copyright notice
	 */
	public static final String COPYRIGHT = "(c) 2024 ZFabrik Software GmbH & Co. KG";


	private static int count;
	public static final String Z2_HOME = "Z2_HOME";
	public static final String TARGET_STATE    = "com.zfabrik.boot.main/process_up";
	
	private ProcessRunnerImpl() {}
	
	public synchronized static int getRunCount() {
		return count;
	}
	
	public synchronized static void start() throws Exception {
		if (count>0) {
			count++;
			return;
		}
		count=1;

		logger.info("Z2 Home Launcher, " + COPYRIGHT);
		
		// start a work unit
		WorkUnit.initCurrent();
		try {
			
			// do some (ugly) init work
			// ----------- touch things ----------
			/*
			 * With some static initializers in the JDK, it is helpful to touch them really early so that 
			 * there initialization happens before anything non-trivial, in particular context class loaders
			 * show up.
			 * 
			 * List:
			 * - JAAS configuration: keeps context class loader in static initialization. We initialize it on the system class path
			 */
			if (System.getProperty("java.security.auth.login.config")!=null) {
				logger.fine("Touching "+Configuration.class.getName());
				Configuration.getConfiguration();
			}
			
			/*
			 * Some proxy basic auth support. The system property com.zfabrik.proxy.auth=basic make the 
			 * process launcher register an authentication handler that respects all <protocol>.proxyUser, <protocol>.proxyPassword
			 */
			
			if ("basic".equalsIgnoreCase(System.getProperty(Foundation.PROXY_AUTH))) {
				Authenticator.setDefault(new ProxyBasicAuthenticator());
			}
			
			File z2Home = getZ2Home();

			Foundation.getProperties().setProperty(Foundation.HOME,z2Home.getAbsolutePath());

			// log and memorize version
			String version="n.a.";
			File versionFile = new File(z2Home,VERSION_FILE);
			if (versionFile.exists()) {
				try (InputStream vin = new FileInputStream(versionFile)) {
					version=new String(IO.readFully(vin),StandardCharsets.UTF_8);
				}
			} else {
                logger.info("No version info found at "+versionFile.getAbsolutePath()+" - installation may be broken");
			}
            Foundation.getProperties().setProperty(Foundation.Z2_VERSION, version);

			//
			// default layout
			//
			Foundation.getProperties().setProperty(Foundation.HOME_LAYOUT_WORK, new File(z2Home,"work").getAbsolutePath());
			Foundation.getProperties().setProperty(Foundation.HOME_LAYOUT_REPOS, new File(z2Home,"work/repos").getAbsolutePath());
			Foundation.getProperties().setProperty(Foundation.HOME_LAYOUT_DATA, new File(z2Home,"data").getAbsolutePath());
			Foundation.getProperties().setProperty(Foundation.HOME_LAYOUT_LOCAL, new File(z2Home,"local").getAbsolutePath());
			Foundation.getProperties().setProperty(Foundation.HOME_LAYOUT_BIN, new File(z2Home,"bin").getAbsolutePath());

			// force determination of instanceid
			Foundation.getInstanceId();

			//
			// crucial first level debug info
			//
			String os = String.format("Using VM v%s by %s at %s on %s (arch: %s)",
					System.getProperty("java.version"),
					System.getProperty("java.vendor"),
					System.getProperty("java.home"),
					System.getProperty("os.name"),
					System.getProperty("os.version"),
					System.getProperty("os.arch")
			);
			logger.info(os);
			String us = String.format("Running Z2 version %s, core build %d as %s in z2 home %s as instance %s, timezone %s, language %s, region %s",
					version,
					Foundation.getCoreBuildVersion(),
					System.getProperty("user.name"),
					new File(System.getProperty(Foundation.HOME)).getCanonicalPath(),
					Foundation.getInstanceId(),
					System.getProperty("user.timezone"),
					System.getProperty("user.language"),
					System.getProperty("user.region")
			);
			logger.info(us);
			
			
			
			// flush all log handlers to make sure the concurrency is resolved cleanly.
			flushHandlers(logger);
			
			if (!Foundation.isWorker() && Foundation.isDevelopmentMode()) {
				logger.info("************************************");
				logger.info("*** Running in DEVELOPMENT mode! ***");
				logger.info("************************************");
			}
						
			// bin is actually not up for discussion
			File z2Run  = new File(z2Home,"bin");

			//
			// determine base config
			//
			// The base config contains bootstrapping config for the repos 
			// as well as several defaults that will all be loaded into 
			// the foundation config
			//
			
			String sCfgFile = System.getProperty(Foundation.CONFIG_FILE, Foundation.CONFIG_FILE_DEF);
			File cfgFile = new File(z2Run,sCfgFile);
			
			if (!cfgFile.exists()) {
				throw new IllegalStateException("Failed to find config file "+cfgFile);
			}
			
			Properties props = new Properties();
			props.load(new FileInputStream(cfgFile));
			// #2055 we do not want to overwrite system properties, but possibly add
			props.keySet().removeAll(Foundation.getProperties().keySet());
			Foundation.getProperties().putAll(props);
			

			// initialize threading system
			WorkManagerImpl.initialize();
			String maxConcurrency = System.getProperty(Foundation.HOME_CONCURRENCY);
			if (maxConcurrency != null) {
				try {
					int maxc = Integer.parseInt(maxConcurrency.trim());
					ApplicationThreadPoolImpl.instance().setMaxConcurrency(maxc);
				} catch (NumberFormatException nfe) {
					throw new Exception("failed to initialize work manager", nfe);
				}
			}
			
			// initialize resource management
			ResourceManagerImpl.initialize();
			
			// initialize components manager.
			ComponentsManager.initialize();

			if (!Foundation.isWorker()) {
				// register home MBean.
				HomeMXBeanImpl.register();
			}
			
			
			// bring the repos up
			Runnable r = IComponentsLookup.INSTANCE.lookup(TARGET_STATE, Runnable.class);
			if (r==null)
				throw new IllegalStateException("Failed to retrieve the repository initialization state ("+TARGET_STATE+")");
			r.run();
			
			
		} finally {
			// complete the init work unit
			WorkUnit.closeCurrent();
		}
		
		// done!
	}

	/**
	 * Identify z2 home
	 */
	public static File getZ2Home() {
		//
		// try identify where we are supposed to run
		//
		String sZ2Home = System.getProperty(Foundation.HOME);
		if (sZ2Home==null) {
			// try environment
			sZ2Home = System.getenv(Z2_HOME);
			if (sZ2Home==null) {
				sZ2Home = "..";
			}
		}
		File z2Home = new File(sZ2Home);
		if (!z2Home.exists()) {
			throw new IllegalArgumentException("Failed to locate z2 home. Tried at "+sZ2Home);
		}
		// create log output folder
		new File(z2Home,"logs").mkdir();
		return z2Home;
	}
	
	private static void flushHandlers(Logger log) {
		if (log!=null) {
			Handler[] handlers = log.getHandlers();
			if (handlers!=null && handlers.length>0) {
				for (Handler h : handlers) {
					h.flush();
				}
			}
			flushHandlers(log.getParent());
		}
	}

	/**
	 * shutdown.
	 */
	public synchronized static void stop() throws Exception {
		_check();
		if (count>1) {
			count --;
			return;
		} 
		try {
			HomeMXBeanImpl.unregister();
		} finally {
			try {
				// shotdown components
				ComponentsManager.shutdown();
			} finally {
				try {
					// shut down the resource system
					ResourceManagerImpl.shutdown();
				} finally {
					try {
						// shut down the work manager
						WorkManagerImpl.shutdown();
					} finally {
						count=0;
					}
				}
			}
		}
	}

	/**
	 * run a task within the thread pool accounting
	 */
	public static void work(final Runnable r) {
		_check();
		// switch to at least this class loading level
		ThreadUtil.cleanContextExecute(
			ProcessRunnerImpl.class.getClassLoader(),
			new Callable<Void>() {
				@Override
				public Void call() throws Exception {
					WorkUnit.work(r);
					return null;
				}
			}
		);
	}

	/**
	 * run a task within the thread pool accounting 
	 */
	public static <T> T work(final Callable<T> c) throws Exception {
		_check();
		// switch to at least this class loading level
		return ThreadUtil.cleanContextExceptionExecute(
			ProcessRunnerImpl.class.getClassLoader(),
			new Callable<T>() {
				public T call() throws Exception {
					return WorkUnit.work(c);
				}
			}
		);
	}
	
	/**
	 * Begin a unit of work. Make your to close the unit of work again. 
	 * Better use the work methods, unless you are really required to 
	 * spread begin and end of a unit of work to different points in the control flow.
	 */
	public static void beginWork(){
		_check();
		WorkUnit.initCurrent();
	}

	/**
	 * End a unit of work.  
	 * Better use the work methods, unless you are really required to 
	 * spread begin and end of a unit of work to different points in the control flow.
	 */
	public static void closeWork(){
		_check();
		WorkUnit.closeCurrent();
	}

	/**
	 * Flag a unit of work as rollback only. I.e. the work will not
	 * be committed at the close of the unit of work but rather rolled
	 * back (if there is any).
	 */
	public static void setRollbackOnly(){
		_check();
		WorkUnit.setRollbackOnlyCurrent();
	}
	
	private static void _check() {
		if (count==0) {
			throw new IllegalStateException("The z2 environment has not been initialized. Call ProcessRunner.start() first.");
		}
	}

	//
	// helper class for BASIC proxy auth
	//
	private static class ProxyBasicAuthenticator extends Authenticator {
		
		protected PasswordAuthentication getPasswordAuthentication() {
			if (getRequestorType() == RequestorType.PROXY && isRequestFromConfiguredProxy()) {
				return new PasswordAuthentication(getProxyUser(), getProxyPassword());
			}
			return null;
		}

		// is the request for the proxy
		private boolean isRequestFromConfiguredProxy() {
			String proxyHost = getRequestingHost();
			String proxyPort = Integer.toString(getRequestingPort());
			return proxyHost.equalsIgnoreCase(getProxyHost()) &&  proxyPort.equalsIgnoreCase(getProxyPort());
		}

		private String getProxyPort() {return getProperty("proxyPort");}
		private String getProxyHost() {return getProperty("proxyHost");}
		private String getProxyUser() {return getProperty("proxyUser");}
		private char[] getProxyPassword() { return getProperty("proxyPassword").toCharArray(); }
		
		// retrieves <protocol>.<suffix> with fallback to <suffix>
		private String getProperty(String suffix) {
			return System.getProperty(getRequestingProtocol().toLowerCase()+"."+suffix,System.getProperty(suffix,""));
		}
	}
	// END
	
	private final static Logger logger = Logger.getLogger(ProcessRunnerImpl.class.getName());

}
