/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.work;

/**
 * Resource that may be registered with a work unit. In general
 * the work unit is responsible of managing the state of such resource.
 * It will inform associated resource about events like begin, rollback, and
 * commit of the resources commit charge (if applicable).
 * After a resource has been closed, it will not be used again. This is the
 * time to release associated internal resources.
 * 
 * Resources of a work unit usually has a very defined life time, like a request cycle.
 * <ol>
 * <li>begin: first interaction during a unit of work</li>
 * <li>possibly: commit cycle
 * 	<ol style="list-style:lower-latin">
 * 	<li>beforeCompletion: All resources before commit</li>
 *  <li>commit: All resources commit</li>
 *  <li>afterCompletion: All resources after commit</li>
 *  </ol>
 * </li>
 * <li>rollback: Invoked if the workunit is rollback only at commit time or actually asked to rollback.</li>
 * <li>close: Always called at the end. For all resources. Regardless of failures.</li>
 * </ol>
 * 
 */
public interface IWorkResource
{
	/**
	 * Start. The resource has been registered with the work unit. Defaults to calling {@link #begin()}
	 */
	default void begin(WorkUnit workUnit) {
		begin();
	};

	/**
	 * Start. The resource has been registered with the work unit.
	 */
	default void begin() {}
	
	/**
	 * The work unit is in the process of completion. No commit or rollback has been called yet. 
	 */
	default void beforeCompletion(boolean rollback) {};

	/**
	 * Commit work 
	 */
	default void commit() {};
		
	/**
	 * Rollback work 
	 */
	default void rollback() {};
	
	/**
	 * Commits have completed. Everything has been successful so far.
	 */
	default void afterCompletion(boolean rollback) {};

	/**
	 * Close. Called in any case
	 */
	default void close() {};
}
