/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.svnaccess;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;

public interface ISvnRepository extends Closeable {

    static long HEAD = -1;

    /**
     * @return
     *  the repository base URL
     */
    String getBaseUrl();

    /**
     * Sets the base URL - this is the z2 component repository URL
     *
     * @param
     *  baseUrl base URL
     */
    void setBaseUrl(String baseUrl);

    /**
	 * Sets the username (like --username)
	 * 
	 * @param
     *  username svn user
	 */
	void setUsername(String username);

    /**
     * Sets the password (like --password)
     *
     * @param
     *  password svn password
     */
	void setPassword(String password);

    /**
     * @return
     *  repository UUID
     *
     * @throws
     *  java.io.IOException
     */
    String getRepositoryUuid() throws IOException;

    /**
     * @return
     * 	current SVNCR revision
     *
     * @throws
     *  java.io.IOException
     */
    long getCurrentCRRevision() throws IOException;

    /**
     * @return
     * 	SVN-Repository root URL - e.g. "http://z2-environment.net/svn/z2-environment/"
     *
     * @throws
     *  java.io.IOException
     */
    String getSvnRootUrl() throws IOException;

    /**
     * @return
     *  CR-path relative to SVN root URL - e.g. "/trunk/z2-base.base"
     *
     * @throws
     *  java.io.IOException
     */
    String getCRPath() throws IOException;

	/**
	 * returns details about a single path
	 *
	 * @param path
	 * 	path relative to the component repository
	 *
     * @param pegRevision
     * 	peg revision, use -1 for HEAD
	 *
	 * @return
	 * 	an {@link SvnDirEntry} containing the details
	 *  or null if the given path does not exist in the given revision
	 *
	 * @throws java.io.IOException
	 */
	SvnInfo info(String path, long pegRevision) throws IOException;

    /**
     * returns details about the base path at HEAD revision
     *
     * @return
     *  most current info of the base-path
     *
     * @throws IOException
     */
    SvnInfo info() throws IOException;

    /**
	 * Retrieves all changes for the given SVN URL and the given revision range and calls the given log entry handler for each change.
	 * If revisionFrom is greater than revisionTo the revision order is descending, otherwise it's ascending. 
	 * 
	 * @param path
	 *	path relative to CR-root
	 *		
	 * @param pegRevision
     * 	peg revision, use -1 for HEAD
	 * 
	 * @param revisionFrom
	 * 	revision from
	 *
	 * @param revisionTo
	 * 	revision to
	 * 
	 * @param logEntryHandler
	 * 	log entry handler is called for each changed resource 
	 *   
   	 * @return
	 * 	number of log entries
	 * 
	 * @throws java.io.IOException
	 * 	if SVN LOG fails
	 */
	int log(String path, long pegRevision, long revisionFrom, long revisionTo, ISvnLogEntryHandler logEntryHandler) throws IOException;
	
	/**
	 * Retrieves all direct children for the given SVN URL and calls the given resource handler for each child
	 * 
	 * @param path
	 *	path relative to CR-root
	 * 
	 * @param pegRevision
	 * 	peg revision, use -1 for HEAD

	 * @param revision
	 * 	revision, use -1 for HEAD
	 *
	 * @param dirEntryHandler
	 * 	resource handler
	 * 
	 * @return
	 * 	number of dir entries
	 * 
	 * @throws java.io.IOException
	 * 	if SVN URL does not exist in the given revision
	 */
	int list(String path, long pegRevision, long revision, IDirEntryHandler dirEntryHandler) throws IOException;
	
	/**
	 * Retrieves the content of the given SVN URL and calls the given stream handler for the input stream
	 * 
	 * @param path
	 * path relative to component repository URL
	 *
     * @param pegRevision
     * 	peg revision, use -1 for HEAD
     * @param revision
	 * 	revision
	 * 
	 * @param
	 * 	streamHandler stream handler
	 * 
	 * @throws java.io.IOException
	 * 	if SVN URL does not exist in the given revision
	 */
	void getContent(String path, long pegRevision, long revision, IStreamHandler streamHandler) throws IOException;
	
	/**
	 * Exports the given directory content to the given target directory. 
	 * The directory will be overridden if it exists already!  
	 * 
	 * @param path
	 * 	path relative to component repository URL
	 * 
	 * @param revision
     * 	revision, use -1 for HEAD (we always use preg revision HEAD)
	 * 
	 * @param targetDir
	 * 	target directory
	 * 
	 * @throws java.io.IOException
	 * 	if SVN URL does not exist in the given revision or targetDir refers to an existing file
	 */
	void export(String path, long revision, File targetDir) throws IOException;
}
