/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.svnaccess;

import static com.zfabrik.impl.svnaccess.PathHelper.concat;
import static com.zfabrik.impl.svnaccess.PathHelper.deSlash;
import static com.zfabrik.impl.svnaccess.PathHelper.deSlashEnd;
import static com.zfabrik.impl.svnaccess.PathHelper.relativize;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.tmatesoft.svn.core.ISVNDirEntryHandler;
import org.tmatesoft.svn.core.ISVNLogEntryHandler;
import org.tmatesoft.svn.core.SVNDirEntry;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.SVNLogEntry;
import org.tmatesoft.svn.core.SVNLogEntryPath;
import org.tmatesoft.svn.core.SVNNodeKind;
import org.tmatesoft.svn.core.SVNURL;
import org.tmatesoft.svn.core.auth.BasicAuthenticationManager;
import org.tmatesoft.svn.core.internal.io.dav.DAVRepositoryFactory;
import org.tmatesoft.svn.core.internal.io.fs.FSRepositoryFactory;
import org.tmatesoft.svn.core.internal.io.svn.SVNRepositoryFactoryImpl;
import org.tmatesoft.svn.core.io.SVNRepository;
import org.tmatesoft.svn.core.io.SVNRepositoryFactory;

import com.zfabrik.impl.svnaccess.svnkit.ApplicationEditor;
import com.zfabrik.svnaccess.IDirEntryHandler;
import com.zfabrik.svnaccess.IStreamHandler;
import com.zfabrik.svnaccess.ISvnLogEntryHandler;
import com.zfabrik.svnaccess.ISvnRepository;
import com.zfabrik.svnaccess.NodeKind;
import com.zfabrik.svnaccess.SvnDirEntry;
import com.zfabrik.svnaccess.SvnInfo;
import com.zfabrik.svnaccess.SvnLogItem;
import com.zfabrik.svnaccess.SvnLogItem.Action;

/**
 * SVNKit based implementation of our Svn Repo abstraction. Requires to have SVNKit on the classpath to load and work.
 * 
 * This implementation will be picked by {@link SvnClientAdapterResource} if an SVNKit library of version 1.7.9 or higher is found - as far as compatibility of
 * SVNKit is established with that version. 
 */
public class SVNKitSvnRepository implements ISvnRepository {
//	private final static Logger LOG = Logger.getLogger(JavaHLSvnRepository.class.getName());
    private String baseUrl;
    private String username,password;

    // fetched on demand
    private SVNRepository repo;
    
    private Long currentRev;
    private String repoUUID;
    private String svnRoot;
    private String crPath;

	@Override
	public void close() throws IOException {
		if (this.repo!=null) {
			try {
				this.repo.closeSession();
			} finally {
				this.repo=null;
			}
		}
	}

	@Override
	public String getBaseUrl() {
		return this.baseUrl;
	}

	@Override
	public void setBaseUrl(String baseUrl) {
		this.baseUrl = deSlash(baseUrl);
	}

	@Override
	public void setUsername(String username) {
		this.username = username;	
	}

	@Override
	public void setPassword(String password) {
		this.password = password;
	}

	@Override
	public String getRepositoryUuid() throws IOException {
		if (this.repoUUID == null) {
			try {
				this.repoUUID = this.repo().getRepositoryUUID(false);
			} catch (SVNException e) {
				throw new IOException(e);
			}
		}
		return this.repoUUID;
	}

	@Override
	public long getCurrentCRRevision() throws IOException {
		if (this.currentRev==null) {
			try {
				this.currentRev = this.repo().getLatestRevision();
			} catch (SVNException e) {
				throw new IOException(e);
			}
		}
		return this.currentRev;
	}

	@Override
	public String getSvnRootUrl() throws IOException {
		if (this.svnRoot==null) {
			try {
				this.svnRoot = deSlashEnd(this.repo().getRepositoryRoot(false).toString());
			} catch (SVNException e) {
				throw new IOException(e);
			}
		}
		return this.svnRoot;
	}

	@Override
	public String getCRPath() throws IOException {
		if (this.crPath==null) {
			this.crPath = relativize(this.baseUrl, getSvnRootUrl());
		}
		return crPath;
	}

	@Override
	public SvnInfo info(String path, long pegRevision) throws IOException {
		SvnInfo i;
		try {
			i = svnInfo(this.repo().info(path,pegRevision));
		} catch (SVNException e) {
			throw new IOException(e);
		}
		if (i==null) {
			throw new IOException(path+" not found");
		}
		return i;
	}

	@Override
	public SvnInfo info() throws IOException {
		return info("",getCurrentCRRevision());
	}

	@Override
	public int log(String path, long pegRevision, long revisionFrom, long revisionTo, final ISvnLogEntryHandler logEntryHandler) throws IOException {
		final int[] count = new int[1];
		
		try {
			this.repo().log(
				new String[]{path},
				revisionFrom,
				revisionTo,
				true,
				false,
				new ISVNLogEntryHandler() {
					@Override
					public void handleLogEntry(SVNLogEntry le) throws SVNException {
						for (Map.Entry<String,SVNLogEntryPath> e : le.getChangedPaths().entrySet()) {
							Action a = null;
							switch (e.getValue().getType()) {
								case SVNLogEntryPath.TYPE_ADDED: a = Action.added; break;
								case SVNLogEntryPath.TYPE_DELETED: a = Action.deleted; break;
								case SVNLogEntryPath.TYPE_MODIFIED: a = Action.modified; break;
								case SVNLogEntryPath.TYPE_REPLACED: a = Action.replaced; break;
							}
							SvnLogItem li = new SvnLogItem(e.getKey(), le.getRevision(), a); 
							try {
								logEntryHandler.handleLogEntry(li);
							} catch (Exception x) {
								throw new RuntimeException(x);
							}
							count[0]++;
						}
					}
				}
			);
		} catch (SVNException e) {
			throw new IOException(e);
		}
		return count[0];
	}

	@Override
	public int list(final String path, long pegRevision, long revision, final IDirEntryHandler dirEntryHandler) throws IOException {
		final List<SvnDirEntry> entries = new LinkedList<SvnDirEntry>();
		try {
			this.repo().getDir(
				deSlash(path),
				pegRevision,
				null,
				new ISVNDirEntryHandler() {
					
					@Override
					public void handleDirEntry(SVNDirEntry de) throws SVNException {
						if (de.getName().length()>0) {
							// must be a real content file
							
							SvnDirEntry di = new SvnDirEntry(
								// local name
								de.getName(), 
								// cr path
								deSlash(concat(path,de.getName())),
								de.getRevision(),
								nodeKind(de.getKind())
							);
							entries.add(di);
						}
						
					}
				}
			);
		} catch (SVNException e) {
			throw new IOException(e);
		}
		for (SvnDirEntry de : entries) {
			try {
				dirEntryHandler.handleSvnDirEntry(de);
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}
		return entries.size();
	}

	@Override
	public void getContent(String path, long pegRevision, long revision, IStreamHandler streamHandler) throws IOException {
		try {
			ByteArrayOutputStream baout = new ByteArrayOutputStream(); 
			this.repo().getFile(
				path,
				pegRevision,
				null,
				baout
			);
			streamHandler.handleStream(new ByteArrayInputStream(baout.toByteArray()));
		} catch (Exception e) {
			throw new IOException(e);
		}
	}

	@Override
	public void export(String path, long pegRevision, File targetDir) throws IOException {
		try {
            SVNURL ol = repo().getLocation();
            SVNURL nl = ol.appendPath(path, false);
            try {
                repo.setLocation(nl, false);
                repo.checkout(pegRevision, null, true, new ApplicationEditor(targetDir));
            } finally {
                repo.setLocation(ol, false);
            }
		} catch (Exception e) {
			throw new IOException(e);
		}
	}


	/**
	 * Turn SVNKit SVNDirEntry into svninfo
	 * @param info
	 * @return
	 */
	private SvnInfo svnInfo(SVNDirEntry info) throws IOException {
		if (info==null) {
			return null;
		}
		String url 		= info.getURL().toString();
		String crpath 	= relativize(url,baseUrl);
		String rootUrl = info.getRepositoryRoot().toString();
		String path     = relativize(url, rootUrl);
				
		return new SvnInfo(
			rootUrl, 
			path, 
			crpath, 
			info.getRevision(), 
			nodeKind(info.getKind()), 
			getRepositoryUuid()
		);
	}

	/**
	 * Turn SVNKit nodekind into nodekind
	 */
    private NodeKind nodeKind(SVNNodeKind kind) {
    	if (SVNNodeKind.DIR.equals(kind)) {
    		return NodeKind.dir;
    	}
    	if (SVNNodeKind.FILE.equals(kind)) {
    		return NodeKind.file;
    	}
    	if (SVNNodeKind.NONE.equals(kind)) {
    		return NodeKind.none;
    	}
    	if (SVNNodeKind.UNKNOWN.equals(kind)) {
    		return NodeKind.unknown;
    	}
    	throw new IllegalArgumentException("Unsupported node kind "+kind);
	}

	/**
     * Create repo session on demand
     */
    private SVNRepository repo() throws IOException {
    	if (this.repo==null) {
    		// take care of all the pre-reqs
            ClassLoader cl = Thread.currentThread().getContextClassLoader();
            Thread.currentThread().setContextClassLoader(null);
            try {
                String proxyHost = System.getProperty("http.proxyHost",System.getProperty("proxyHost"));
                String proxyPort = System.getProperty("http.proxyPort",System.getProperty("proxyPort"));
                String url = this.baseUrl;
                if (url.indexOf("://")<0) {
                    url = "file:///"+new File("").toURI().toString().substring("file:/".length())+url;
                }
                SVNURL svnurl = SVNURL.parseURIEncoded(url);
				if (url.startsWith("svn:")) {
                    SVNRepositoryFactoryImpl.setup();
                    this.repo = SVNRepositoryFactory.create(svnurl);
                } else if (url.startsWith("http:")) {
                    DAVRepositoryFactory.setup();
                    this.repo = DAVRepositoryFactory.create(svnurl);
                } else if (url.startsWith("https:")) {
                    DAVRepositoryFactory.setup();
                    this.repo = DAVRepositoryFactory.create(svnurl);
                } else if (url.startsWith("file:")) {
                    FSRepositoryFactory.setup();
                    this.repo = FSRepositoryFactory.create(svnurl);
                } else
                    throw new IOException("SVNKit SVN access: don't know how to handle \""+url+"\"");

				// we configure an auth manager. The default uses the current OS user's settings. We definitely do not want that.
				//
		        BasicAuthenticationManager authManager = new BasicAuthenticationManager(this.username,this.password);

		        // configure proxy use for SVNKit
		        if (proxyHost!=null && (proxyHost=proxyHost.trim()).length()>0 && proxyPort!=null && (proxyPort=proxyPort.trim()).length()>0) {
	                int pp;
	                try {
                        pp = Integer.parseInt(proxyPort.trim());
	                } catch (NumberFormatException nfe) {
                        throw new RuntimeException("Failed to configure SVNKit proxy settings - invalid port number \""+proxyPort+"\"",nfe);
	                }
	                authManager.setProxy(proxyHost,pp, System.getProperty("proxyUser"), System.getProperty("proxyPassword"));
		        }

		        this.repo.setAuthenticationManager(authManager);
		        
            } catch (SVNException svne) {
            	throw new IOException(svne);
            } finally {
            	Thread.currentThread().setContextClassLoader(cl);
            }
    	}
        return this.repo;
    }
    
}
