/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.svnaccess;

import java.util.logging.Logger;

import com.zfabrik.resources.provider.Resource;
import com.zfabrik.svnaccess.ISvnRepository;

/**
 * Svn-Client-Adapter supports javahl and svnkit (commandline is not implemented yet)!
 * 
 * This is NOT based on the SVN Client Adapter by tigris.org.
 *
 * This z2 resource uses the JhlClientAdapterFactory for both javahl and svnkit: svnkit provides an own svnkit-javahl
 * implementation with the standard javahl API.
 *
 * So it depends only(!) on the classpath whether the Apache javahl.jar is used (which is a wrapper for the
 * native svn libraries) or whether the svnkit-javahl.jar is used (which delegates all javahl API methods to the
 * java-based svnkit implementation).
 *
 */
public class SvnClientAdapterResource extends Resource {

    private static final Logger logger = Logger.getLogger(SvnClientAdapterResource.class.getName());

    public static enum Mode {
    	/**
    	 * Auto detect at first use. We prefer SVNKit and if that is not found, Java HL will be used
    	 */
    	DETECT, 
    	/**
    	 * Use JavaHL
    	 */
    	JAVAHL, 
    	/**
    	 * Use the SVNKit Subversion Client library
    	 */
    	SVNKIT, 
    	/**
    	 * None found. Access not functional.
    	 */
    	NONE 
    };
    
    // detected once 
    private static Mode detected = Mode.DETECT; 
    
    // determined for the instance
    private Mode mode = Mode.DETECT;
    
    public SvnClientAdapterResource() {}
    
    /**
     * For testing
     */
    public SvnClientAdapterResource(Mode mode) {
    	this.mode = mode;
    }
    
    
    @Override
    public <T> T as(Class<T> clz) {
        if (ISvnRepository.class.equals(clz)) {
            _load();
            switch (mode) {
            case JAVAHL:
            	return clz.cast(new JavaHLSvnRepository());
            case SVNKIT:
            	return clz.cast(new SVNKitSvnRepository());
            default: 
    			throw new IllegalStateException("No repository access method out of SVNKit and JavaHL found. SVN Repository access is not supported. Please either install and configure JavaHL on your OS platform or install SVNKit with your Z2 system. Check the how-tos on the z2 wiki.");
            }
        }
        return super.as(clz);
    }

    private void _load() {
    	if (mode==Mode.DETECT) {
	    	synchronized (SvnClientAdapterResource.class) {
				mode = detected;
		    	if (mode==Mode.DETECT) {
		    		// determine operational mode.
		    		try {
		        		// if we find an SVNKit class, we prefer that
		    			Class.forName("org.tmatesoft.svn.core.io.SVNRepository",false,SvnClientAdapterResource.class.getClassLoader()); 
		    			logger.info("Found SVNKit support. Will use SVNKit repository access.");
		    			mode = Mode.SVNKIT;
		    		} catch (ClassNotFoundException e) {
		    			// no SVNKit. 
		    			try {
		            		// Test for JavaHL
		        			Class.forName("org.apache.subversion.javahl.SVNClient",false,SvnClientAdapterResource.class.getClassLoader()); 
		        			logger.info("Found JavaHL support. Will use JavaHL repository access.");
		        			mode = Mode.JAVAHL;
		    			} catch (ClassNotFoundException f) {
		        			logger.info("JavaHL not found. Not using JavaHL repository access.");
		    			}
		    		}
		        	if (mode==Mode.DETECT) {
		        		mode = Mode.NONE;
		        	}
	    			detected = mode;
		    	}
	    	}
    	}
    	if (mode==Mode.NONE) {
    		throw new IllegalStateException("No repository access method out of SVNKit and JavaHL found. SVN Repository access is not supported. Please either install and configure JavaHL on your OS platform or install SVNKit with your Z2 system");
    	}
    }
}
