/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.gitcr;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;

import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.Transport;
import org.eclipse.jgit.transport.TransportProtocol;
import org.eclipse.jgit.transport.URIish;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;

import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.components.provider.fs.AbstractFileSystemComponentRepository;
import com.zfabrik.components.provider.fs.FileSystemImpl;
import com.zfabrik.components.provider.fs.IAbstractFileSystem;
import com.zfabrik.components.provider.fs.MultiRootFSComponentRepositoryDB;
import com.zfabrik.impl.gitcr.helper.GitCommand;
import com.zfabrik.util.runtime.Foundation;

/**
 * Git based components repository based on {@link AbstractFileSystemComponentRepository}
 * <p>
 * The Git CR clones the source repo (i.e. the one defined by <code>gitcr.path</code>) into the z2-repo cache.
 * On sync the cloned repo fetches all changes from the source repo, so that git does all the ugly stuff :-)
 *
 * <p>
 * Known GitCR properties:
 * <table>
 * <tr><td><code>gitcr.uri</code></td><td>The URI to the source repo. Can be an absolute path, a local path relative to bin, or a remote URL</td></tr>
 * <tr><td><code>gitcr.priority</code> or <code>gitcr.prio</code></td><td>The priority of the repository</td></tr>
 * <tr><td><code>gitcr.branch</code></td><td>The branch to clone. Defaults to the active branch of the source repository</td></tr>
 * <tr><td><code>gitcr.ref</code></td><td>
 *  A Git ref to fetch and check out as active content. This can be a tag (<code>refs/tags/&lt;tag name&gt;</code>), or a branch
 * (<code>refs/heads/&lt;branch name&gt;</code>) or HEAD for the default branch. 
 * For compatibility, it can also be a remote tracking branch (<code>refs/remotes/origin/&lt;branch name&gt;</code> or <code>origin/&lt;branch name&gt;</code>).
 * <p>
 * As this information is used to compute a ref spec, this may not be a single commit, as git fetch requires some form of ref.
 * </p>
 * <p>
 * In particular this can be "HEAD" to simply follow the remote default branch!
 * </p>
 * Use this option, if you need more control than given by <code>gitcr.branch</code>. Note that the latter will be used with preference and hence needs to be omitted, if you want to use <code>gitcr.ref</code>.
 * </td></tr>
 * <tr><td><code>gitcr.roots</code></td><td>Comma-separated list of paths relative to the repo root to search for modules. Can be left empty for the default, which is searching from the root</td></tr>
 * <tr><td><code>gitcr.depth</code></td><td>Fetch depth on clone. Corresponds to --depth parameter. Set to 1 for shallow clone</td></tr>
 * </table>
 */
public class GitComponentRepositoryImpl extends AbstractFileSystemComponentRepository {
	
	private static final String Z2_HEAD = "refs/heads/_Z2_HEAD_";

	private static final String REFS_HEADS_PREFIX = "refs/heads/";

	private static final String REFS_REMOTES_ORIGIN_PREFIX = "refs/remotes/origin/";

	private static final String ORIGIN_PREFIX = "origin/";

	/**
	 * A comma separated list of folders in the repo to search for components. Can be left empty, if the repo
	 * is only at top level. Otherwise a list in order of decreasing priority can be given.
	 */
	public final static String GITCR_ROOTS  = "gitcr.roots";

	/**
	 * URI of the git source repository.
	 */
	public final static String GITCR_URI  = "gitcr.uri";

	/**
	 * Priority of the repository within the repository chain. See {@link IComponentsRepository}.
	 * Both gitcr.priority and gitcr.prio are valid keys.
	 */
	public final static String GITCR_PRIORITY = "gitcr.priority";
	public final static String GITCR_PRIO = "gitcr.prio";

	/**
	 * Branch to clone from the source repository. Setting this is equivalent to setting {@link #GITCR_REF} with the branch name prefixed by <code>refs/heads/</code>.
	 */
	public final static String GITCR_BRANCH = "gitcr.branch";

	/**
	 * A Git ref to fetch and check out as active content. This can be a tag (<code>refs/tags/&lt;tag name&gt;</code>), or a branch
	 *  (<code>refs/heads/&lt;branch name&gt;</code>) or HEAD for the default branch. 
	 * For compatibility, it can also be a remote tracking branch (<code>refs/remotes/origin/&lt;branch name&gt;</code> or <code>origin/&lt;branch name&gt;</code>).
	 * <p>
	 * As this information is used to compute a ref spec, this may not be a single commit, as git fetch requires some form of ref.
	 * </p>
	 * <p>
	 * In particular this can be "HEAD" to simply follow the remote default branch!
	 * </p>
	 * Use this option, if you need more control than given by <code>gitcr.branch</code>. Note that the latter will be used with preference and hence needs to be omitted, if you want to use <code>gitcr.ref</code>.	 
	 */
	public final static String GITCR_REF = "gitcr.ref";

	/**
	 * User name for authentication at the repository
	 */
	public final static String GITCR_USER = "gitcr.user";

	/**
	 * Password for authentication at the repository
	 */
	public final static String GITCR_PWD  = "gitcr.password";

	/**
	 * If set to 'true', the gitcr will be ignore silently if the defined gitcr.uri is invalid.
	 * Defaults to 'true', in which case only a warning is logged, when the origin repository is not reachable.
	 *
	 * @Deprecated Please use {@link IComponentsRepository#COMPONENT_REPO_MODE}
	 */
	public final static String GITCR_OPTIONAL = "gitcr.optional";

	/**
	 * network timeout-value in seconds. Defaults to 10seconds.
	 */
	public final static String GITCR_TIMEOUT = "gitcr.timeout";
	
	/**
	 * Git clone depth. Corresponds to "--depth" parameter
	 */
	public final static String GITCR_DEPTH = "gitcr.depth";
	
	
	
	/**
	 * The original Git-Repo
	 */
	private URIish originUri;

	/**
	 * Credentials
	 */
	private UsernamePasswordCredentialsProvider credentials;

	/**
	 * The ref spec for fetching. We use the same ref name for source and target (!). That is, we might have
	 * <ul>
	 * <li><p>+HEAD:HEAD</p> for the case of useDefaultBranch</li> 
	 * <li><p>+refs/tags/&lt;sometag&gt;:refs/tags/&lt;sometag&gt;<p>, if gitcr.ref references a tag, or</li>
	 * <li><p>+refs/heads/&lt;branch&gt;:refs/heads/&lt;branch&gt;<p>, if gitcr.ref references a branch</li>
	 * </ul>
	 * In particular, we do not maintain remote branches per se, as we checkout the exact same ref anyway and we do never ever push anything or
	 * commit from the local repo.
	 */
	private RefSpec fetchRefSpec;

	/**
	 * true <=> this repository is optional
	 */
	private boolean isOptional;

	/**
	 * network timeout
	 */
	private int timeout;

	/**
	 * network timeout
	 */
	private Integer depth;

	/**
	 * actual git repo folder
	 */
	private File localRepo;
	
	/**
	 * Expected repository configuration. These properties are used to
	 * compare for repo purging (see #1879)
	 */
	private Properties expectedConfiguration;

	public GitComponentRepositoryImpl(String name, Properties props) {
		super(
			name,
			-1, // dummy prio
			Integer.MAX_VALUE / 2, // max depth
			false // no auto configure
		);

		// compute expected config
		this.expectedConfiguration = new Properties();
		this.expectedConfiguration.putAll(props);
		
		// remove some config that we consider insignificant for repo purging
		this.expectedConfiguration.remove(GITCR_OPTIONAL);
		this.expectedConfiguration.remove(GITCR_PRIO);
		this.expectedConfiguration.remove(GITCR_PRIORITY);
		this.expectedConfiguration.remove(GITCR_BRANCH);
		this.expectedConfiguration.remove(GITCR_REF);
		this.expectedConfiguration.remove(GITCR_ROOTS);
		this.expectedConfiguration.remove(GITCR_DEPTH);

		// configure (you simply need to know that this must happen before the rest but after expected config)
		configure(getPrio(name, props, 500));
		
		// get folder inside the repo-cache z2/work for the cloned git repository.
		File cacheRoot = getCacheRoot();
		// the actual local repo
		this.localRepo = new File(cacheRoot, GIT_CLONE_DIR);

		// determine roots
		Map<String,IAbstractFileSystem> roots = new LinkedHashMap<String, IAbstractFileSystem>();
		for (String r : getRoots(name,props)) {
			roots.put(r,new FileSystemImpl(new File(this.localRepo,r)));
		}
		setRoots(roots);

		// do we need a connection to the origin repository?
		this.isOptional = getIsOptional(name, props, true) || super.isRelaxedMode();

		// get the original git repo from the props
		this.originUri = getOrigRepo(name, props, this.isOptional);

		// get the branch to clone
		String branch = getBranchToClone(name, props);
		String ref;
		if (branch!=null) {
			ref = REFS_HEADS_PREFIX+branch;
		} else {
			// get ref to use
			ref = getRef(name, props);
		}
		// check some consistency
		if (ref==null) {
			throw new IllegalStateException("Unless support of the remotes default branch is configured (it isn't), you need to specify either "+GITCR_BRANCH+" or "+GITCR_REF+" to make gitcr work: "+name);			
		}
		
		String targetRef;
		if ("HEAD".equals(ref)) {
			// we use Z2_HEAD when fetching from HEAD, as Git can be picky about fetching to HEAD
			targetRef = Z2_HEAD;
		} else {
			targetRef = ref;
		}
		
		// add the "+" to force even non-fast-forward updates
		this.fetchRefSpec = new RefSpec("+"+ref+":"+targetRef);
		logger.fine("Using fetch reference " +ref);

		// get the credentials for (remote) repositories
		this.credentials = getCredentials(name, props);

		this.timeout = getTimeout(name, props, 10);
		this.depth = getDepth(name, props);

		if (!Foundation.isWorker()) {
			logger.info("Using " + this.toString());
		}
	}
	
	@Override
	protected Properties getExpectedConfiguration() {
		return this.expectedConfiguration;
	}

	/**
	 * Scans for changes in this GitCR which implies:
	 * <ul><li>if <code>current</code> is <code>null</code> the source repository will be cloned into the repository cache</li>
	 * 	   <li>if the repository
	 */
	public MultiRootFSComponentRepositoryDB scan(MultiRootFSComponentRepositoryDB current) {
		logger.fine("scanning " +getLabel());
		
		// configure command
		GitCommand gc = new GitCommand(this.getName(), originUri, localRepo, fetchRefSpec, isOptional, credentials, timeout, depth);
		
		if (current == null || ! this.localRepo.exists()) {
			// no DB exists yet, must clone the source repository
			gc.doClone();
		} else {
			// DB already exists, so try to fetch the deltas from the origin. If this fails create a new clone
			gc.doFetch();
		}
		// AbstractFileSystemComponentRepository.scan() performs the the real scan upon clone's working directory
		return super.scan(current);
	}

	public File getLocalRepo() {
		return localRepo;
	}

	@Override
	public String toString() {
		return getLabel();
	}

	public String getLabel() {
		return "GIT-CR: " 
				+"origin:"	+this.originUri
				+ ",ref:" + this.fetchRefSpec 
				+ ",depth:" + this.depth 
				+ ",optional:"+this.isOptional
			    +", " + super.toString();
	}

	// -- private ---------------------------------------------------------------------------------

	private static String normalize(String in) {
		if (in!=null) {
			in = in.trim();
			if (in.length()>0) {
				return in;
			}
		}
		return null;
	}

	private static String firstNotBlank(String... strs) {
		for (String s : strs) {
			String sTrim = normalize(s);
			if (sTrim != null) {
				return sTrim;
			}
		}
		return null;
	}


	/*
	 * Returns roots
	 */
	private static List<String> getRoots(String name, Properties props) {
		String rs = props.getProperty(GITCR_ROOTS);
		if (rs==null || (rs=rs.trim()).length()==0) {
			return Arrays.asList("");
		}
		List<String> result = new ArrayList<>();
		for (String r : rs.split(",")) {
			if ((r=r.trim()).length()>0) {
				result.add(r);
			}
		}

		logger.fine("read GitCR " +name+ "::" +GITCR_ROOTS+ " = " +result);
		return result;
	}

	/*
	 * Returns whether this gitcr is an optional repository (i.e. an invalid gitcr.uri property value will be ignored).
	 */
	private static boolean getIsOptional(String name, Properties props, boolean defValue) {
		boolean result = false;
		if (Foundation.isDevelopmentMode()) {
			// http://redmine.z2-environment.net/issues/911: allow optional repositories only in dev-mode
			String isOptional = normalize(props.getProperty(GITCR_OPTIONAL));
			result = isOptional==null? defValue: Boolean.parseBoolean(isOptional);
		}
		logger.fine("read GitCR " +name+ "::" +GITCR_OPTIONAL+ " = " +result);
		return result;
	}

	/*
	 * Returns the priority of the repository by reading 'gitcr.prio' from the z.properties.
	 */
	private static int getPrio(String name, Properties props, int defValue) {
		int prio;
		String prios = firstNotBlank(props.getProperty(GITCR_PRIO), props.getProperty(GITCR_PRIORITY));
		if (prios == null) {
			prio=defValue;
		} else {
			try {
				prio = Integer.parseInt(prios);
			} catch (NumberFormatException nfe) {
				throw new IllegalStateException("GIT-CR '" + name + ": Invalid priority ("+GITCR_PRIO+") specification (" + prios + "): " + name, nfe);
			}
		}

		logger.fine("read GitCR " +name+ "::" +GITCR_PRIO+ " = " +prio);
		return prio;
	}

	/*
	 * Returns the URI to the source repository by reading 'gitcr.uri' from the z.properties
	 */
	private static URIish getOrigRepo(String name, Properties props, boolean isOptional) {
		String uri = normalize(props.getProperty(GITCR_URI));
		logger.fine("read GitCR " +name+ "::" +GITCR_URI+ " = " +uri);

		if (uri==null) {
			throw new IllegalStateException("Missing property " +GITCR_URI+ " in " +name);
		}

		URIish uriish;
		try {
			uriish = new URIish(uri);
		} catch (URISyntaxException e1) {
			throw new IllegalStateException("GIT-CR '" + name + ": '" + uri + "' is not a valid Git-URI");
		}

		// try normalize to work relative to Z2 home
		if (!uriish.isRemote()) {
			try {
				File f = new File(uri);
				if (!f.isAbsolute()) {
					// relative!
					String zHome = normalize(System.getProperty(Foundation.HOME));
					if (zHome != null) {
						f = new File(new File(zHome),uri);
					} else {
						f = new File(uri);
					}

				}
				uri = f.getCanonicalPath();
				uriish = new URIish(uri);
			} catch (Exception e1) {
				throw new IllegalStateException("GIT-CR '" + name + ": '" + uri + "' is not a valid Git-URI");
			}
		}

		if (isOptional) {
			// skip all checks for optional repositories
			return uriish;
		}

		boolean canHandle = false;
		for (TransportProtocol transp : Transport.getTransportProtocols()) {
			if (transp.canHandle(uriish)) {

				if (! canHandle && transp.getSchemes().contains("file")) {
					// do some checks in advance
					File gitDir = new File(uri);
					String absPath = null;
					try {
						absPath = gitDir.getCanonicalPath();
					} catch (IOException e) {
						throw new IllegalStateException("GIT-CR '" +name+ ": The path " +gitDir+ " defined in " +name+ "::" +GITCR_URI+ " cannot be canonicalized!", e);
					}

					if (! gitDir.exists()) {
						throw new IllegalStateException("GIT-CR '" +name+ ": The path " +gitDir+ " (abs-path: "+absPath+ ") defined in " +name+ "::" +GITCR_URI+ " does not exists!");
					}
					if (! gitDir.isDirectory()) {
						throw new IllegalStateException("GIT-CR '" +name+ ": The path " +gitDir+ " (abs-path: "+absPath+ ") defined in " +name+ "::" +GITCR_URI+ " is not a directory!");
					}
					if (! gitDir.canRead()) {
						throw new IllegalStateException("GIT-CR '" +name+ ": The path " +gitDir+ " (abs-path: "+absPath+ ") defined in " +name+ "::" +GITCR_URI+ " cannot be accessed! Please check permissions.");
					}
				}

				canHandle = true;
			}
		}

		if (! canHandle) {
			throw new IllegalStateException("GIT-CR '" + name + ": The uri " +uri+ " defined in " +GITCR_URI+ " cannot be handled by this git implementation!");
		}

		return uriish;
	}

	/*
	 * Returns the ref to use
	 */
	public static String getRef(String name, Properties props) {
		String ref = normalize(props.getProperty(GITCR_REF));
		// here are some compatibility cases:
		if (ref!=null) {
			if (ref.startsWith(ORIGIN_PREFIX)) {
				// this is origin/branch - we reduce to branch.
				ref = REFS_HEADS_PREFIX+ref.substring(ORIGIN_PREFIX.length());
			} else
			if (ref.startsWith(REFS_REMOTES_ORIGIN_PREFIX)) {
				// this is refs/remotes/origin/branch - we reduce to branch.
				ref = REFS_HEADS_PREFIX+ref.substring(REFS_REMOTES_ORIGIN_PREFIX.length());
			}
			logger.fine("Using GitCR " +name+ "::" +GITCR_REF+ " = " +ref);
		}
		return ref;
	}

	/*
	 * Returns the branch to clone from the source repository by reading 'gitcr.branch' from the z.properties.
	 */
	private static String getBranchToClone(String name, Properties props) {
		String branch = normalize(props.getProperty(GITCR_BRANCH));
		logger.fine(()->"read GitCR " +name+ "::" +GITCR_BRANCH+ " = " +branch);
		return branch;
	}

	/*
	 * Returns a JGit-Credentials object containing the property-values for 'gitcr.user' and 'gitcr.password'.
	 * Defaults to null, if no user and password are defined.
	 */
	private static UsernamePasswordCredentialsProvider getCredentials(String name, Properties props) {
		UsernamePasswordCredentialsProvider result = null;

		String user = normalize(props.getProperty(GITCR_USER));
		if (user != null) {
			String passwd = normalize(props.getProperty(GITCR_PWD));
			if (passwd == null) {
				throw new IllegalStateException("GIT-CR '" + name + " has definded " + GITCR_USER + ", but " + GITCR_PWD + " is missing!");
			}

			result = new UsernamePasswordCredentialsProvider(user, passwd);
		}

		logger.fine(()->"read GIT-CR " +name+ "::" +GITCR_USER+ " = " +user);
		return result;
	}

	private int getTimeout(String name, Properties props, int defValue) {
		String timeout = normalize(props.getProperty(GITCR_TIMEOUT));
		int result = timeout==null? defValue: Integer.parseInt(timeout);

		logger.fine(()->"read GIT-CR " +name+ "::" +GITCR_TIMEOUT+ " = " +result);
		return result;
	}
	
	private Integer getDepth(String name, Properties props) {
		String depth = normalize(props.getProperty(GITCR_DEPTH));
		if (depth!=null) {
			logger.fine(()->"read GIT-CR " +name+ "::" +GITCR_DEPTH+ " = " +depth);
			return Integer.parseInt(depth);
		} else {
			return null;
		}
	}

	/* the name of the folder containing the cloned repository */
	private final static String GIT_CLONE_DIR = "git";

	private final static Logger logger = Logger.getLogger(GitComponentRepositoryImpl.class.getName());
}
