/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.gitcr.helper;

import java.io.File;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.URIish;

import com.zfabrik.components.provider.util.LockingRevFile;
import com.zfabrik.impl.gitcr.GitComponentRepositoryImpl;
import com.zfabrik.util.fs.FileUtils;
import com.zfabrik.util.runtime.Foundation;

/**
 * The actual GitCR operations in complete for better testability
 */
public class GitCommand {
	private final static Logger logger = Logger.getLogger(GitComponentRepositoryImpl.class.getName());

	
	private File localRepo;
	private URIish originUri;
	private boolean isOptional;
	private RefSpec fetchRefSpec;
	private CredentialsProvider credentials;
	private int timeout;
	private String toString;
	private Integer depth;
	
	public GitCommand(String toString, URIish originUri, File localRepo, RefSpec fetchRefSpec, boolean isOptional, CredentialsProvider credentials, int timeout, Integer depth) {
		super();
		this.toString = toString;
		this.originUri = originUri;
		this.localRepo = localRepo;
		this.fetchRefSpec = fetchRefSpec;
		this.isOptional = isOptional;
		this.credentials = credentials;
		this.timeout = timeout;
		this.depth = depth;
	}
	
	/*
	 * clone the repository and clean-up on failure
	 */
	public void doClone() {
		try (GitLock gl = new GitLock(this.localRepo)) {
			logger.info(()->"Cloning "+this.toString()+" using "+this.fetchRefSpec+" from " + this.originUri);
			// purging the folder allows to call clone in any situation
			FileUtils.delete(this.localRepo);
			// clone the origin repository into the z2 work folder
			try (Repository clonedRepo = GitTools.cloneRepository(
					this.originUri, 
					this.localRepo, 
					this.fetchRefSpec, 
					this.credentials, 
					this.timeout,
					this.depth
			)) {
				// check out FETCH_HEAD
				String sha = GitTools.checkOutRef(clonedRepo,this.fetchRefSpec.getDestination());
				logger.info("After cloning "+this.fetchRefSpec.getSource()+" check out is at "+sha);
			}
		} catch (Exception e) {
			// delete any repository ruins
			FileUtils.delete(this.localRepo);
			logger.log(Level.WARNING, "GIT-CR: Failed cloning " + this.toString(),e);
			// something went wrong - maybe we are offline.
			if (this.isOptional) {
				logger.info("GIT-CR: '" + this.originUri.getHumanishName() + "' will be ignored because it's an optional repository or running in relaxed mode.");

			} else {
				throw new CloneFailedException(toString(),this.localRepo.getAbsolutePath(), e);
			}
		}
	}

	/**
	 * fetch the deltas and try clone on failure
	 */
	public void doFetch() {
		if (Foundation.isOfflineMode()) {
			logger.info("GIT-CR: '" + this.originUri.getHumanishName() + "' will be ignored (Running offline)");
			return;
		}
		logger.info("Fetching "+this.toString()+" using "+this.fetchRefSpec+" from " + this.originUri);
		try (
			GitLock gl = new GitLock(this.localRepo);
		    Repository repo = open();
		) {
			// fetch
			GitTools.fetchRepository(repo, this.fetchRefSpec, credentials, timeout, this.depth);
			// update working copy
			String sha = GitTools.checkOutRef(repo,this.fetchRefSpec.getDestination());
			logger.info("After fetching "+this.fetchRefSpec.getSource()+" check out is at "+sha);
		} catch (Exception e) {

			// fetching failed!

			// just log a short description
			logger.log(Level.WARNING, "GIT-CR: Updating " + this.toString() + " failed with exception",e);

			if (this.isOptional) {
				// origin is not reachable, but repository is optional (maybe we are offline). Log this incident and try to continue
				logger.info("GIT-CR: '" + this.originUri.getHumanishName() + "' will be ignored because it's an optional repository or running in relaxed mode");

			} else {
				// throw exception in production mode
				throw new FetchFailedException(toString(), e);
			}
		}
	}

	private Repository open() throws IOException {
		FileRepositoryBuilder b = new FileRepositoryBuilder();
		return b.setGitDir(new File(this.localRepo, ".git")).build();
	}
	
	@Override
	public String toString() {
		return "GIT-CR: "+this.toString;
	}

	/**
	 * As Git can cause problems with non-removed lock files after crashes, we use a 
	 * safe FileLock to lock z2 around any issued Git command and simply delete any git lock 
	 * file found before running the actual Git operation.
	 * (see #2061)
	 */
	private static class GitLock implements AutoCloseable {
		private LockingRevFile lock;
		
		public GitLock(File localRepo) throws IOException {
			// attempt to lock on ${localRepo}/../gitcr.lock
			File gitcrLock = new File(localRepo.getParentFile(),"gitcr.lock");
			logger.fine("Locking on lock file "+gitcrLock);
			lock = new LockingRevFile(gitcrLock);
			// open and lock
			lock.open();
			// check for a Git lock file and remove it if found
			File meta = new File(localRepo, ".git");
			File gitLock = new File(meta,"index.lock");
			logger.fine("Checking on Git lock file "+gitLock);
			if (gitLock.exists()) {
				logger.warning("Deleting left-over Git index lock "+gitLock.getName());
				gitLock.delete();
			}
		}
		
		@Override
		public void close() throws Exception {
			if (this.lock!=null) {
				this.lock.close();
			}
		}
	}
	
}
