/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.gitcr.helper;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.logging.Logger;

import org.eclipse.jgit.api.FetchCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.ResetCommand.ResetType;
import org.eclipse.jgit.lib.ConfigConstants;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.RemoteConfig;
import org.eclipse.jgit.transport.URIish;

/**
 * Provides some useful static helpers like
 * {@link #isOriginReachable(URIish)}, {@link #isValidRepository(URIish)}, {@link #cloneRepository(URIish, File, CredentialsProvider, int)}, ...
 */
public class GitTools {
	private static final String ORIGIN = "origin";

	private final static Logger LOG = Logger.getLogger(GitTools.class.getName());

	/**
	 * Clones the given remote repository into the given destination folder. The method clones all branches but doesn't perform a checkout.
	 *   
	 * @param remoteUri URI of the remote repository
	 * @param destFolder local destination folder
	 * @param credentials user credentials
	 * @return the cloned repository
	 * @throws IOException if something went wrong
	 */
	public static Repository cloneRepository(URIish remoteUri, File destFolder, RefSpec fetchSpec, CredentialsProvider credentials, int timeout, Integer depth) throws IOException {

		// For http://redmine.z2-environment.net/issues/902:
		// split clone into its piece in order to get the chance to set "core.autocrlf"
		try (Git git = Git.init().
					setBare(false).
					setDirectory(destFolder).
					call()) {
			
			Repository repo = git.getRepository();
			
			// setting "core.autocrlf=false" helps to solve http://redmine.z2-environment.net/issues/902
			StoredConfig config = repo.getConfig();
			config.setString(ConfigConstants.CONFIG_CORE_SECTION, null, ConfigConstants.CONFIG_KEY_AUTOCRLF, String.valueOf(false));
			
			// add origin - clone all branches
			RemoteConfig remoteCfg = null;
			try {
				remoteCfg = new RemoteConfig(config, ORIGIN);
			} catch (URISyntaxException e) {
				throw new IOException("Failed to configure origin repository", e);
			}
			remoteCfg.addURI(remoteUri);
			remoteCfg.update(config);
			config.save();
			
			long tStart = System.currentTimeMillis();
			
			// we use a dedicated fetch ref spec - so we do actually only fetch the 
			// branch we care about. Or HEAD.
			FetchCommand cmd = git.
				fetch().
				setProgressMonitor(new ProgressMonitorImpl(LOG)).
				setRemote(ORIGIN).
				setCredentialsProvider(credentials).
				setTimeout(timeout).
				setRefSpecs(fetchSpec);
			if (depth!=null) {
				cmd.setDepth(depth);
			}
			cmd.call();
	
			LOG.info("Clone completed within " + (System.currentTimeMillis() - tStart) + "msec");
			
//			git.getRepository().getRefDatabase().getRefs("").forEach((n,v)->System.err.println(n+" - "+v));

			return repo;
		} catch (Exception e) {
			throw new IOException("Failed to clone repository at " + destFolder.getAbsolutePath()+" from "+remoteUri, e);
		}
	}

	/**
	 * Checks out the given ref branch in the given repository. 
	 * @param targetBranch a local or remote branch name.
	 * @throws IOException if something went wrong
	 */
	public static String checkOutRef(Repository repo, String ref) throws IOException {
		try (Git git = new Git(repo)){
			Ref checkout = git
			.checkout()
			.setName(ref)
			.call();
			
			LOG.info("checkout: "+checkout);
			// In order to force on update from the branch in the repo
			// to the workspace - in particular, if we have already checked out
			// before, we force a reset. This also makes non-fast-forward work!
			Ref reset = git.reset()
			.setMode(ResetType.HARD)
			.call();
			
			LOG.info("reset: "+reset);
			
			Ref head = git.getRepository().getRefDatabase().findRef("HEAD");
			LOG.info("head: "+head);
			
			// show where we are
			return head.getObjectId().getName();
		} catch (Exception e) {
			throw new IOException("Failed to check out ref '" + ref + "' inside " + repo + "!",e);
		}
	}
	

	/**
	 * Fetch deltas from a source-repository    
	 */
	public static void fetchRepository(Repository repo, RefSpec fetchSpec, CredentialsProvider credentials, int timeout, Integer depth) throws Exception {
		// clone already exists. fetch the deltas
		long tStart = System.currentTimeMillis();
		try (Git git = new Git(repo)) {
			// get the deltas from the origin repository
			FetchCommand cmd = git.
			fetch().
			setRefSpecs(fetchSpec).
			setProgressMonitor(new ProgressMonitorImpl(LOG)).
			setCredentialsProvider(credentials).	// set user/password if defined - can be null
			setTimeout(timeout);
			if (depth!=null) {
				cmd.setDepth(depth);
			}
			cmd.call();
			LOG.info("Fetched deltas within " + (System.currentTimeMillis() - tStart) + "msec");
		}
	}

		
}
