/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.java;

import com.zfabrik.util.runtime.Foundation;

/**
 * Supported language level abstraction
 */
public enum LangLevel { 
	/**
	 * Java 11 as of JDK 11.x
	 */
	JAVA11("11"), 
	/**
	 * Java 12 as of JDK 12.x
	 */
	JAVA12("12"),
	/**
	 * Java 13 as of JDK 13.x
	 */
	JAVA13("13"), 
	/**
	 * Java 14 as of JDK 15.x
	 */
	JAVA14("14"),
	/**
	 * Java 15 as of JDK 15.x
	 */
	JAVA15("15"); 

	public static LangLevel HIGHEST = JAVA15;
	
	private String value;

	LangLevel(String value) {
        this.value = value;
    }
    
    @Override
    public String toString() {
    	return value;
    }
    
    /**
	 * The current language level is taken from a system property or by inspecting the runtime. This method implements
	 * the official way. The current default is {@value #HIGHEST}.
	 */
	public static LangLevel determine() {
		String ls = System.getProperty(Foundation.LANGUAGE_LEVEL,"").trim();
		if (ls.length()>0) {
			LangLevel ll = parse(ls);
			if (ll==null) {
				throw new IllegalStateException("Unsupported system property value: "+Foundation.LANGUAGE_LEVEL+"="+ls);
			}
			return ll;
		}
		String vs = System.getProperty("java.version");
		if (vs!=null) {
			if (vs.startsWith("15")) {
				return JAVA15;
			} 
			if (vs.startsWith("14")) {
				return JAVA14;
			} 
			if (vs.startsWith("13")) {
				return JAVA13;
			} 
			if (vs.startsWith("12")) {
				return JAVA12;
			} 
			if (vs.startsWith("11")) {
				return JAVA11;
			} 
			// check for unsupported versions
			if (
				vs.startsWith("9") ||
				vs.startsWith("1.")
			) {
				throw new IllegalStateException("Java version "+vs+" is not supported anymore by this release of z2");
			}
		}
		return HIGHEST;
	}

	public static LangLevel parse(String value) {
    	for (LangLevel l : LangLevel.values()) {
    		if (l.value.equals(value)) {
    			return l;
    		}
    	}
    	return null;
    }

}