/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.states;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IDependencyComponent;

/**
 * System states are abstract target configurations of a z2 environment at runtime.
 * <p>
 * System states are declared as components of type <code>com.zfabrik.systemState</code>. 
 * <p>
 * The most important use-case of this is the heterogeneous designation of worker processes to different purposes 
 * using the system state mechanism. In essence, the runtime should only download resources of component and execute components as 
 * far as required for the implemented scenario. The system state mechanism allows to define scenario 
 * requirements via a graph of system state and component dependencies. See also the worker process javadoc.
 * <p>
 * System states can be <em>attained</em>. Attaining a system state translates to <em>preparing</em> (see {@link IDependencyComponent}) all
 * components that declare a participation in the state using the configuration property {@link #STATES_PARTICIPATION} or by declaring 
 * {@link #STATES_DEPENDENCY} on the state. 
 * <p>
 * System states implement {@link IDependencyComponent}. When preparing, all dependencies will be prepared as well. This happens during any verification and 
 * synchronization so that system state's will be re-enforced at that point in time. 
 * <p>
 * For example, the system state <code>com.zfabrik.boot.main/process_up</code> is declared like this:
 * <pre>
 * #
 * # a system state
 * #
 * com.zfabrik.component.type=com.zfabrik.systemState
 * #
 * # it's a generalization that applies to worker processes
 * # as well as home processes. For example, the distributed
 * # component provisioning participates in this state to advertise
 * # itself at the package management
 * #
 * com.zfabrik.systemStates.participation=\
 * 	   com.zfabrik.boot.main/home_up,\
 * 	   com.zfabrik.boot.main/worker_up
 * #
 * #
 * </pre>
 *  
 * @author hb
 *
 */

public interface ISystemState {
	/**
	 * Type of a system state component.
	 */
	String TYPE = "com.zfabrik.systemState";
	/**
	 * State components declare dependency on other components by specifying a comma-separated list of component names. 
	 * All components will be queried for implementations of {@link IDependencyComponent} and will be invoked when the
	 * state component gets prepared, which is in particular 
	 * <ul>
	 * <li>upon statisfaction of component dependencies (see {@link IComponentDescriptor#DEPENDENCIES}</li>
	 * <li>at all verifications/synchronizations for states that are or are depending on home or worker target states</li>
	 * </ul>  
	 * State dependencies will therefore be revisited recurringly. This make system states a mechanism to (re-) enforce
	 * system health.
	 */
	final static String STATES_DEPENDENCY = "com.zfabrik.systemStates.dependency";
	
	/**
	 * Components declare participation in a state components life cycle by specifying a comma-separated list of component names.
	 * A component participated with, requires all participating components to be prepared and will be dependendant on its
	 * participating components. This is in effect simply the same dependency impact as achieved by a {@link #STATES_DEPENDENCY} declaration
	 * but declared on the other end of the vector.  
	 */
	static final String STATES_PARTICIPATION = "com.zfabrik.systemStates.participation";


}
