/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.hubcr;

import java.io.File;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.provider.util.AbstractComponentRepository;
import com.zfabrik.components.provider.util.FSCRDBComponent;
import com.zfabrik.hubcr.RemoteComponentRepositoryDB;
import com.zfabrik.util.runtime.Foundation;

/**
 * A remote repository hub component repository. This repository connects to a Z2 hosted remote
 * repository that provides optimized repository content in the sense that no local compilation
 * is required anymore and that no source code will be transmitted over the wire.
 * Typically the hub repository is the only one used aside from the local pre-built core repository.
 *
 * <p/>
 * Component properties that have a meaning:
 * <table>
 * <tr><td><code>hubcr.url</code></td><td>The URL to the source repo. Typically this is of the form http://&lt;server&gt;:&lt;port&gt;/z_hubcr</td></tr>
 * <tr><td><code>hubcr.timeout</code></td><td>Connection timeout in ms for connecting to the providing side (defaults to 10000)</td></tr>
 * <tr><td><code>hubcr.user</code></td><td>The remote user to use (optional but required by default)</td></tr>
 * <tr><td><code>hubcr.password</code></td><td>The remote user's password</td></tr>
 * <tr><td><code>hubcr.priority</code></td><td>The priority of the repository (defaults to 500)</td></tr>
 * </table>
 * <p/>
 * In addition there is a system property <code>com.zfabrik.hubcr.mode</code> that if set to <code>relaxed</code> (as opposed to the default of <code>strict</code>)
 * supports relaxed behavior if remote database updates do not succeed. If relaxed, the repository will continue with the latest
 * known database. Assuming component resources have been fetched previously, this allows to survive short outages gracefully.
 *
 * @author hb
 *
 */

public class ComponentRepositoryImpl extends AbstractComponentRepository<RemoteComponentRepositoryDB> {
	private static final String HUBCR_URL = "hubcr.url";
	private static final String HUBCR_USER = "hubcr.user";
	private static final String HUBCR_PASSWORD = "hubcr.password";
	private static final String HUBCR_PRIORITY = "hubcr.priority";
	private static final String HUBCR_TIMEOUT = "hubcr.timeout";
	private static final String HUBCR_MODE = "com.zfabrik.hubcr.mode";
	private static final String HUBCR_MODE_RELAXED = "relaxed";
	private static final String HUBCR_MODE_STRICT = "strict";

	private String name,url,user;
	private long timeout;
	private RemoteClient remoteClient;
	private boolean relaxed;

	public ComponentRepositoryImpl(String name, Properties properties) {
		super(name, RemoteComponentRepositoryDB.class);
		this.name = name;
		this.url = properties.getProperty(HUBCR_URL,"").trim();
		if (url.length()==0) {
			throw new IllegalArgumentException("Must specify property "+HUBCR_URL+": "+this.name);
		}

		String prios = properties.getProperty(HUBCR_PRIORITY,"").trim();
		if (prios.length() > 0) {
			try {
				super.configure(Integer.parseInt(prios));
			} catch (NumberFormatException nfe) {
				throw new IllegalStateException("Invalid priority ("+HUBCR_PRIORITY+") specification (" + prios + "): " + name, nfe);
			}
		} else {
			super.configure(500);
		}

		String timeouts = properties.getProperty(HUBCR_TIMEOUT,"").trim();
		if (timeouts.length() > 0) {
			try {
				this.timeout = Long.parseLong(timeouts);
			} catch (NumberFormatException nfe) {
				throw new IllegalStateException("Invalid timeout ("+HUBCR_TIMEOUT+") specification (" + timeouts+ "): " + name, nfe);
			}
		} else {
			this.timeout = 10000;
		}

		this.user = properties.getProperty(HUBCR_USER,"").trim();
		if (user.length()==0) {
			user = null;
		}
		String password=null;
		if (user!=null) {
			password = properties.getProperty(HUBCR_PASSWORD,"").trim();
		}

		String mode = System.getProperty(HUBCR_MODE,HUBCR_MODE_STRICT).trim();
		this.relaxed = mode.equals(HUBCR_MODE_RELAXED);

		if (!Foundation.isWorker()) {
			logger.info("Using " + this.toString());
			if (this.relaxed) {
				logger.warning("HUB Component Repository running in relaxed mode. Do not use in production!");
			}
		}
		this.remoteClient = new RemoteClient(this.url,this.user,password,this.timeout);
	}

	@Override
	public void download(FSCRDBComponent component, File folder) {
		if (component.isHasFolder()) {
			long start = System.currentTimeMillis();
			logger.fine("Downloading component \""+component.getName()+"\"... : "+this.name);
			try {
				remoteClient.download(component,folder);
				start = System.currentTimeMillis()-start;
				logger.info("Completed download of component \""+component.getName()+"\" after "+start+"ms: "+this.name);
			} catch (Exception e) {
				throw new RuntimeException("Failed to download component "+component.getName()+": "+this.name, e);
			}
		}
	}

	public RemoteComponentRepositoryDB scan(RemoteComponentRepositoryDB current) {
		Long rev = (current!=null? current.getRevision() : null);
		try {
			checkOfflineMode();
			RemoteComponentRepositoryDB db = remoteClient.fetchDBUpdate(rev);
			if (db==null) {
				// we are still up to date
				return current;
			}
			return db;
		} catch (Exception e) {
			if (current!=null && this.relaxed) {
				logger.log(Level.WARNING,"Caught exception during scan ("+e.toString()+"). Will continue with latest known due to running in relaxed mode.");
				return current;
			} else {
				throw new RuntimeException("Failed to scan for repo updates: "+this.name, e);
			}
		}
	};

	@Override
	public String toString() {
		return "HUB-CR: " +
			super.toString() +
			",url:" + this.url +
			",username:" + this.user+
			",mode:"+(this.relaxed? HUBCR_MODE_RELAXED:HUBCR_MODE_STRICT)+
			",timeout:"+this.timeout;
	}


	private final static Logger logger = Logger.getLogger(ComponentRepositoryImpl.class.getName());
}
