/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.compiler;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.java.IJavaComponent;
import com.zfabrik.components.java.JavaComponentUtil;
import com.zfabrik.components.java.build.ICompilationContext;
import com.zfabrik.components.java.build.ICompiler;
import com.zfabrik.components.java.build.ICompilerContext;
import com.zfabrik.resources.provider.Resource;

public class AllCompiler extends Resource implements ICompiler {

	@Override
	public <T> T as(Class<T> clz) {
		if (clz.equals(ICompiler.class)) {
			return clz.cast(this);
		} else {
			return super.as(clz);
		}
	}

	public void init(ICompilerContext context) {}
	
	@Override
	public boolean compile(ICompilationContext compilationContext) {
		long tStart = System.currentTimeMillis();

		String compName = compilationContext.getComponentName();
		File[] src = compilationContext.getSourceFolders();
		File dest = compilationContext.getOutputFolder();
		ClassLoader cl = compilationContext.getClassPath();
		IJavaComponent.Part part = compilationContext.getPart();
		
		assert (compName != null);
		assert (src != null);
		assert (dest != null && dest.isDirectory());
		assert (cl != null);

		boolean success = true;

		// retrieve the list of compilers required by the target component
		List<String> compileOrder = _getCompileOrder(compName, part);

		if (logger.isLoggable(Level.FINE)) {
			logger.fine("Compile " + Arrays.asList(src) +" for part "+part + " of "+ compName + " using " + compileOrder + " compiler" + ((compileOrder.size() > 1)? "s": ""));
		}

		// optimize single language case
		
		for (String compilerID : compileOrder) {
			ICompiler compiler = _getCompilerComponentByID(compilerID);
			logger.fine("Run " + compilerID + " compiler on " + compName);
			// compile
			success = compiler.compile(compilationContext);
			if (! success) {
				// stop if one compiler fails
				return false;
			}
			
		}
		long deltaT = System.currentTimeMillis() - tStart;
		perfLog.fine("Compilation of part "+part+ " of " + compName + " took " + deltaT + "msec");
		return success; 
	}


	// -- private ---------------------------------------------------------------------------------
	
	/*
	 * Returns the compile order for the given component.
	 * If no order is defined the method returns [java]
	 */
	private List<String> _getCompileOrder(String compName, IJavaComponent.Part part) {
		List<String> result = new ArrayList<String>();
		
		IComponentDescriptor comp = IComponentsLookup.INSTANCE.lookup(compName, IComponentDescriptor.class);
		if (comp == null) throw new InternalError("Cannot find component " + compName);
	
		String order = null;
		
		// look for part specific order
		switch (part) {
		case PUBLIC: 
			order = comp.getProperty(IJavaComponent.PUBLIC_COMPILE_ORDER);
			break;
		case PRIVATE: 
			order = comp.getProperty(IJavaComponent.PRIVATE_COMPILE_ORDER);
			break;
		case TEST: 
			order = comp.getProperty(IJavaComponent.TEST_COMPILE_ORDER);
			break;
		}
		if (order==null) {
			// default
			order = comp.getProperty(IJavaComponent.COMPILE_ORDER);
		}
		
		if (order == null) {
			// no compile-order defined -> use java
			result.add("java");
			
		} else {
			StringTokenizer loopOrder = new StringTokenizer(order, ",");
			while (loopOrder.hasMoreTokens()) {
				result.add(loopOrder.nextToken().trim());
			}
		}
		
		return result; 
	}

	/*
	 * Retrieves the compiler component for the given compiler ID as an ICompiler.
	 * Throws exceptions if there's no such compiler or if there are more than one 
	 * or the compiler doesn't implement ICompiler
	 */
	private ICompiler _getCompilerComponentByID(String compilerID) {
		ICompiler result;
		try {
			// find the adequate compiler component: 
			String compilerComponentName = JavaComponentUtil.getCompilerComponentById(compilerID);
			result = IComponentsLookup.INSTANCE.lookup(compilerComponentName, ICompiler.class);
			
			if (result == null) {
				throw new IllegalStateException("Component " + compilerComponentName + " pretends to be a compiler but it does not provide " + ICompiler.class.getName());
			}

		} catch (IOException e) {
			throw new RuntimeException("Failed to find compiler for " + compilerID, e);
		}
		
		return result;
	}

	private final static Logger logger = Logger.getLogger(AllCompiler.class.getName());
	private final static Logger perfLog = Logger.getLogger("Performance." + AllCompiler.class.getName());
}
