/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.java.IJavaComponent;
import com.zfabrik.components.java.build.ICompiler;
import com.zfabrik.util.fs.FileUtils;

public class BuildHelper {

	public static FileFilter ALL_NON_JAVA_FILES = new FileFilter() {
		public boolean accept(File pathname) {
			String name = pathname.getName();
			return ((pathname.isDirectory()) || (!name.endsWith(".java"))) && (!name.startsWith("."));
		}
	};

	public static FileFilter ALL_FILES = new FileFilter() {
		public boolean accept(File pathname) {
			return true;
		}
	};
	
	public static FileFilter ALL_BUT_HIDDEN = new FileFilter() {
		public boolean accept(File pathname) {
			return ! pathname.getName().startsWith(".");
		}
	};

	//
	// ------------------------------------------
	//

	public static void delete(File g) {
		if (g.isDirectory()) {
			for (File f : g.listFiles()) {
				delete(f);
			}
		}
		g.delete();
	}


	private static void _jarIt(File base, JarOutputStream jo, FileFilter filter) throws Exception {
		// pass 1: only folders
		_jarIt(base, "", jo, filter, true);
		// pass 2: files
		_jarIt(base, "", jo, filter, false);
	}

	private static void _jarIt(File base, String current, JarOutputStream jo, FileFilter filter, boolean folders) throws Exception {
		if (filter == null)
			filter = ALL_FILES;
		if (base.isDirectory()) {
			if (folders) {
				_addFolderToJAR(base, current, jo);
			}
			String path;
			for (File f : base.listFiles(filter)) {
				path = current.length()==0? f.getName() : current + "/" + f.getName();
				// recurse
				_jarIt(f, path, jo, filter, folders);
			}

		} else {
			if (!folders) {
				_addFileToJAR(base, current, jo);
			}
		}
	}

	private static void _addFolderToJAR(File f, String path, JarOutputStream jo) throws IOException {
		// do not add "/"
		if (path.length()>0) {
			path = path + "/"; // append to make it appear as a folder
			JarEntry je = new JarEntry(path);
			if (f != null)
				je.setTime(f.lastModified());
			je.setSize(0);
			jo.putNextEntry(je);
			jo.closeEntry();
		}
	}

	private static void _addFileToJAR(File f, String path, JarOutputStream jo) throws IOException {
		JarEntry je = new JarEntry(path);
		je.setTime(f.lastModified());
		int len = (int) f.length();
		byte[] buffer = new byte[len];
		// read that file completely
		InputStream fin = new FileInputStream(f);
		int l, s = 0;
		while ((l = fin.read(buffer, s, len - s)) > 0)
			s += l;
		fin.close();
		jo.putNextEntry(je);
		jo.write(buffer);
		jo.closeEntry();
	}

	public static void compileAndJar(String cname,File instanceFolder, File[] srcs,File gen,File jar,ClassLoader cl,	ICompiler compiler, IJavaComponent.Part part) {
		// only consider src folders that exist
		ArrayList<File> sfs = new ArrayList<File>(srcs.length);
		for (File src : srcs) {
			if (src.exists()) {
				sfs.add(src);
			}
		}
		if (sfs.isEmpty()) {
			return;
		}
		srcs = sfs.toArray(new File[sfs.size()]);

		boolean failed;
		try {
			if (!gen.exists()) {
				gen.mkdirs();
			}
			failed = !compiler.compile(new CompilationContextImpl(cname, instanceFolder, srcs, gen, cl, part));
			if (!failed) {
				if (!jar.getParentFile().exists()) {
					jar.getParentFile().mkdirs();
				}
				// jar it up!
				JarOutputStream jo = null;
				try {
					// Include source files into jars for eclipsoid
					for (File src : srcs) {
						FileUtils.copy(src, gen, ALL_BUT_HIDDEN);
					}
					
					// check for a manifest in gen.
					File mf = new File(gen,"/META-INF/MANIFEST.MF");
					if (!mf.exists()) {
						// create an artificial manifest (why... I do not remember... :-( )
						Manifest man = new Manifest();
						Attributes at = man.getMainAttributes();
						at.put(Attributes.Name.MANIFEST_VERSION, "1.0");
						jo = new JarOutputStream(new FileOutputStream(jar), man);
					} else {
						// just jar it with the given Manifest file
						jo = new JarOutputStream(new FileOutputStream(jar));
					}
					logger.finer("JARring " + jar.getName());
					_jarIt(gen, jo, ALL_FILES);
				} finally {
					if (jo!=null) {
						jo.close();
					}
				}
			}
		} catch (Throwable e) {
			logger.log(Level.SEVERE, "Component Build failed with error: "+cname,e);
			if (e instanceof VirtualMachineError) {
				throw (VirtualMachineError)e;
			}
			throw new RuntimeException("Component Build failed with error: "+cname,e);
		}
		if (failed) {
			throw new RuntimeException("Build failed (see compiler messages): " + cname);
		}
	}

	private final static Logger logger = ComponentsBuilder.logger;
}
