/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java.jdt;
import java.io.File;
import java.io.FileOutputStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jdt.internal.compiler.ClassFile;
import org.eclipse.jdt.internal.compiler.CompilationResult;
import org.eclipse.jdt.internal.compiler.ICompilerRequestor;

/**
 * Output from the compiler. The CompilerReqestor is to take what the compiler throws at it.
 * This can be compilation results and errors.
 * 
 * @author hb
 */

public class CompilerRequestorImpl implements ICompilerRequestor {
	private File outFolder;
	private boolean hasError;

	public CompilerRequestorImpl(File outFolder) {
		this.outFolder = outFolder;
	}

	public void acceptResult(CompilationResult result) {
		if (result.hasProblems()) {
			IProblem[] problems = result.getProblems();
			for (IProblem problem : problems) {
				if (problem.isWarning()) {
					if (logger.isLoggable(Level.FINEST)) {
						logger.warning(String.format("Compile Warning %s",problemToText(problem)));
					}
				} else if (problem.isError()) {
					this.hasError = true;
					logger.severe(String.format("\n\nCompile Error %s\n",problemToText(problem)));
				}
			}
		}

		if (!this.hasError) {
			ClassFile[] classFiles = result.getClassFiles();
			for (ClassFile classFile : classFiles) {
				String classFileName = NameEnvironmentImpl.concat(classFile.getCompoundName(),'/') + ".class";

				if (logger.isLoggable(Level.FINEST)) {
					logger.finest("Writing class file "+classFileName+" into "+this.outFolder);
				}

				byte[] bytes = classFile.getBytes();
				File outFile = new File(this.outFolder, classFileName);
				if (!outFile.getParentFile().exists()) {
					outFile.getParentFile().mkdirs();
				}
				FileOutputStream fout = null;
				try {
					try {
						fout = new FileOutputStream(outFile);
						fout.write(bytes);
					} finally {
						fout.close();
					}
				} catch (Exception e) {
					throw new RuntimeException("Failed to write .class file: " + classFileName, e);
				}
			}
		}
	}
	
	private String problemToText(IProblem problem) {
		return String.format("(%d): %s at %s:%d", 
				problem.getID(),
				problem.getMessage(), 
				new String(problem.getOriginatingFileName()), 
				problem.getSourceLineNumber()
		);
	}
	
	public boolean hasError() {
		return this.hasError;
	}

	private final static Logger logger = Logger.getLogger(CompilerRequestorImpl.class.getName());
	
}