/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java.jdt;
import java.io.File;
import java.io.InputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.eclipse.jdt.internal.compiler.classfmt.ClassFileReader;
import org.eclipse.jdt.internal.compiler.classfmt.ClassFormatException;
import org.eclipse.jdt.internal.compiler.env.INameEnvironment;
import org.eclipse.jdt.internal.compiler.env.NameEnvironmentAnswer;

import com.zfabrik.components.java.JavaComponentClassLoader;

/**
 * Input to the jdt compiler. A name env provides classes to link to, sources to compile...
 */
public class NameEnvironmentImpl implements INameEnvironment {
	private final static Logger LOG = Logger.getLogger(NameEnvironmentImpl.class.getName());
	private ClassLoader cl;
	private File[] sourceFolders;
	private Map<String,File> sfnToFile = new HashMap<String, File>();
	private String encoding;
	private Set<String> packages = new HashSet<>();

	public NameEnvironmentImpl(ClassLoader cl, String encoding, File... sourceFolders) {
		this.cl = cl;
		this.sourceFolders = sourceFolders;
		this.encoding = encoding;
		this.preloadPackages(this.cl);
	}

	public void cleanup() {
		this.sfnToFile.clear();
		this.packages.clear();
	}

	public NameEnvironmentAnswer findType(char[][] compoundName) {
		return _findType(concat(compoundName,'/'));
	}

	public NameEnvironmentAnswer findType(char[] typeName, char[][] packageName) {
		return _findType(getName(packageName, typeName, '/'));
	}

	/*
	 * The eclipse compiler checks, if a name (derived from using a symbol in code) is a package.
	 * This can be unexpected stuff like java.MyClass, mypackage.String.
	 * 
	 * On the other hand, if we claim java.MyClass IS a package, it may stop looking deeper for
	 * MyClass (which is actually in the same package as the referencing class).
	 * 
	 * Hence we may not easily claim something is a package, if it actually is a class.
	 *
	 * And then, the system and/or the platform class loaders do not answer ClassLoader.getDefinedPackage meaningfully.
	 */
	@Override
	public boolean isPackage(char[][] parentCompoundName, char[] packageName) {
		String pname = getName(parentCompoundName, packageName, '.');
		String sname = pname.replace('.','/');
		
		if (this.packages.contains(sname)) {
			return true;
		}
		boolean r;
		// simple check: Does it contain a hyphen (this includes stuff like package-info)
		if (pname.indexOf('-')>=0) {
			r = false;
		} else {
			// check in our sources: If we have the folder (this is not 100% according to JLS (7.4ff) but... come on. 
			if (Optional.ofNullable(_findFile(sname)).map(File::isDirectory).orElse(false)) {
				r = true;
			} else {
				// Otherwise, we use a simple heuristics: If the last segment starts with an upper case character, we assume it's NOT a package
				// (this captures the jre class loader cases effectively)
                r = !Character.isUpperCase(packageName[0]);
			}
		}
		log(()->(r? "is a ":"not a ")+"package: "+pname);
		// nothing passed: Assume it is not a package 
		return r;
	}

	// load all packages from the classpath (except JDK)
	private void preloadPackages(ClassLoader cl) {
		if (cl==null) {
			return;
		}
		if (cl instanceof JavaComponentClassLoader) {
			JavaComponentClassLoader jcl = (JavaComponentClassLoader)cl;
			jcl.getClassPathPackages().forEach(p->this.packages.add(p));
			jcl.getParents().forEach(this::preloadPackages);
		}
	}

	private String getName(char[][] parentCompoundName, char[] packageName, char separator) {
		String slashedClassName = (parentCompoundName==null || parentCompoundName.length==0? 
			new String(packageName) :
			new StringBuilder().append(concat(parentCompoundName, separator)).append(separator).append(packageName).toString()
		);
		return slashedClassName;
	}

	// -------

	public static String concat(char[][] compoundName, char separator) {
		if (compoundName==null) {
			return "";
		}
		StringBuilder b = new StringBuilder(200);
		for (char[] s : compoundName) {
			if (b.length() > 0) {
				b.append(separator);
			}
			b.append(s);
		}
		return b.toString();
	}

	private File _findFile(String name) {
		File f = this.sfnToFile.get(name);
		if (f!=null) {
			return f;
		}
		if (this.sourceFolders != null) {
			for (File sf : this.sourceFolders) {
				f = new File(sf, name);
				if (f.exists()) {
					this.sfnToFile.put(name,f);
					return f;
				}
			}
		}
		return null;
	}
	
	private File _findSourceFile(String slashedClassName) {
		return _findFile(slashedClassName+".java");
	}

	private NameEnvironmentAnswer _findType(String slashedClassName) {
		// try as source file first
		File sf = _findSourceFile(slashedClassName);
		if (sf != null) {
			return new NameEnvironmentAnswer(new FileCompilationUnit(sf,this.encoding), null);
		}
		// try as byte array next
		String rn = slashedClassName + ".class";
		InputStream in = this.cl.getResourceAsStream(rn);
		if (in != null) {
			try {
				try {
					return new NameEnvironmentAnswer(ClassFileReader.read(in, rn, true), null);
				} finally {
					in.close();
				}
			} catch (Exception e) {
				if (e instanceof ClassFormatException) {
					int ec = ((ClassFormatException)e).getErrorCode();
					throw new RuntimeException("Failed to read class file (error "+ec+"): " + rn, e);
				}
				throw new RuntimeException("Failed to read class file " + rn, e);
			}
		}
		return null;
	}

	private void log(Supplier<String> s) {
		LOG.finest(s);
	}
}
