/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.processing;

import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.provider.IComponentDescriptorProcessor;
import com.zfabrik.components.provider.props.Evaluator;

/**
 * A jexl3 component descriptor processor. This processor is based on the Apache Commons JEXL library
 * version 3.0 (see <a href="https://commons.apache.org/proper/commons-jexl">https://commons.apache.org/proper/commons-jexl</a>).
 * <p>
 * All values of properties tagged as of JEXL3 processing style in the original component descriptor source will be considered JEXL expressions.
 * If evaluating to a non-null value, the expression evaluation result will be set as resulting property
 * value of the same property name. If the expression evaluates to <code>null</code>, the corresponding property
 * will not be set.
 * </p>
 * <p>
 * The following implicit variables are made available by this processor:
 * <table>
 * <tr><th>Variable name</th><th>Definition</th><th>Example</th></tr>
 * <tr><td>system</td><td>{@link System#getProperties()}</td><td>`user dir is ${system["user.dir"]}`</td></tr>
 * <tr><td>env</td><td>{@link System#getenv()}</td><td>"JAVA_HOME is ${env.JAVA_HOME}"</td></tr>
 * <tr><td>this</td><td>The actual property set in its evaluated form. This may be used to resolve properties that are again computed by some processor. Note that a max evaluation depth of 50 is enforced when resolving properties defined in the property set.</td><td>this['com.zfabrik.component.type']</td></tr>
 * <tr><td>components</td><td> {@link IComponentsManager#INSTANCE}</td><td>components.getComponent("mymodule/mycomponent").getProperty("myProp")</td></tr>
 * </table>
 * </p>
 * Using the <code>this</code> variable, re-use of definitions within a single descriptor is possible. For example, assuming
 * some other processor <b>ext</b> is involved, a property set
 * <pre><code>
 * hostName\:JEXl3=env.hostName
 * database\:ext=_THE_DB_
 * ds.prop.url\:JEXl3=`jdbc:derby://${this.hostName}/${this.database};create=true`
 * </code></pre>
 * would be evaluated resolving across processors.
 * <p>
 * Note that a JEXL expression does not necessarily evaluate to a string object.
 * Neither is the input property value to be processed considered a JEXL string. For example, the expression
 * <pre>
 * `Hallo this is ${system["user.name"]}`
 * </pre>
 * does indeed evaluate to a string. As does
 * <pre>
 * "Hallo this is ${system[\"user.name\"]}"
 * </pre>
 * (without processing the ${} expression) and more advanced
 * <pre>
 * system["os.name"].startsWith("Linux")? "We are running on a Linux OS" : "We are running on something else"
 * </pre>
 * but not
 * <pre>
 * 500+10
 * </pre>
 * which is indeed an integer.
 * </p>
 * See also <a href="https://commons.apache.org/proper/commons-jexl/reference/syntax.html">https://commons.apache.org/proper/commons-jexl/reference/syntax.html</a>
 * for more syntax information.
 *
 * @author hb
 *
 */
public class Jexl3ComponentDescriptorProcessor implements IComponentDescriptorProcessor {
	
	/**
	 * Style processed by this processor
	 */
	public final static String JEXL3_STYLE = "JEXL3";

	@Override
	public Evaluator createEvaluator() {
		return new Jexl3ComponentDescriptorEvaluator();
	}
}
