/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.java;

import java.io.File;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.util.runtime.Foundation;

/**
 * Client interface of a Java component. I.e. any Java component provides an implementation of this interface
 * when looked up.
 * <p>
 * A Java component is declared by using the component type <code>com.zfabrik.java</code> (see also {@link IComponentDescriptor#COMPONENT_TYPE})
 * <p>
 * Java components generally serve Java type definitions to other components, for example Web applications. 
 * <p>
 * One important defaulting mechanism applies to Java components: By convention, other component types that require or support an implementation, for example a Web application will use {@link JavaComponentUtil} to
 * identify a Java component that is used to load types from. Given a project <code>com.acme.myproject</code>, the default Java component to look for is <code>com.acme.myproject/java</code>, i.e. the
 * project name with an appended "/java". Components within the project use the private loader of that Java component by convention.
 * <p>
 * See also {@link IComponentsRepository} for a typical component repository layout.
 * <p>
 * In most cases Java components do hold Java resources, e.g. Java source files or JAR libraries. In some cases all types required are provided elsewhere, so that the project itself does not
 * hold any Java resources. In that case a Java component does not require a folder structure but may be specified only a set of properties defining references to other Java components.
 * <p>
 * The folder structure of Java components assigns meaning to the following files and folders:
 * <dl>
 * <dt>z.properties</dt>
 * <dd>The component descriptor. Typically this looks like
 * <pre>
 * com.zfabrik.component.type=com.zfabrik.java
 * java.publicReferences=&lt;java component 1&gt;, &lt;java component 2&gt;,...
 * java.privateReferences=&lt;java component n&gt;, &lt;java component n+1&gt;,...
 * </pre> 
 * 
 * 
 * </dd>
 * <dt>src.api</dt>
 * <dd>Public type definitions that will be compiled and exposed so that they are visible by other Java components. See {@link IJavaComponent#PUBREFS}.</dd>
 * <dt>bin.api/lib</dt>
 * <dd>JAR libraries to be added to the public types of the Java Component.</dd>
 * <dt>bin.api/classes</dt>
 * <dd>Pre-compiled Java code, i.e. <code>.class</code> files and other resources to be added to the public types of the Java Component.</dd>
 * <dt>src.impl</dt>
 * <dd>Type definitions that will be compiled and made available for use within the project (see above) but will not be exposed to other Java components. See {@link IJavaComponent#PRIREFS}.</dd>
 * <dt>bin.impl/lib</dt>
 * <dd>JAR libraries to be added to the private types of the Java Component.</dd>
 * <dt>bin.impl/classes</dt>
 * <dd>Pre-compiled Java code, i.e. <code>.class</code> files and other resources to be added to the private types of the Java Component.</dd>
 * <dt>src.test</dt>
 * <dd>Type definitions that will be compiled and made available for use just like the types defined in src.impl. In contrast to src.impl, the source files in src.test 
 * will be ignored unless the runtime runs in development mode, that is the system property {@link Foundation#MODE} is set to "development".</dd>
 * <dt>bin.test/lib</dt>
 * <dd>JAR libraries to be added to the private types of the Java Component iff running in development mode.</dd>
 * <dt>bin.test/classes</dt>
 * <dd>Pre-compiled Java code, i.e. <code>.class</code> files and other resources to be added to the private types of the Java Component iff running in development mode.</dd>
 * </dl>
 * 
 * <p>
 * Java components may "include" components of type <code>com.zfabrik.files</code>. This can be useful for libraries that cannot be shared by
 * just re-using the classes but instead require to be used within the using component's context, e.g. to load classes by name that would not be within 
 * the library's scope. This is true for older versions of the Spring framework for example.
 * <p>
 * Files components may be included as API or impl provisioning depending on the reference used. The included source files or pre-compiled binaries are
 * expected to comply to the following folder structure:
 * <p>
 *   
 * <dl>
 * <dt>src</dt>
 * <dd>Source files that require compilation before execution.</dd>
 * <dt>bin/lib</dt>
 * <dd>JAR libraries</dd>
 * <dt>bin/classes</dt>
 * <dd>Pre-compiled Java code, i.e. <code>.class</code> files and other resources</dd>
 * </dl>
 * 
 * <p>
 * The following properties can be used in the component descriptor:
 * <dl>
 * 
 * <dt>java.publicReferences</dt>
 * <dd>
 * Points to other java components whose types will be shared with this one (and maybe others).
 * <p>
 * Everything referenced as public reference will be visible to the public interface of the referencing component as well as to all 
 * referencing the referencing component. In other words: References are transitive.
 * In particular, anything required to compile the public types of a Java component must be referenced via this reference property.
 * <p>
 * Components may be specified as a comma-separated list. Component names that have no "/" will be defaulted by appending "/java".
 * <p>
 * Alternatively this reference points to a <code>com.zfabrik.files</code> component that must have a bin folder that will be included into this java component's java resources.
 * The files resource may also have a <code>src</code> folder that will be copied befor compilation into the public interface in <code>src.api</code> or <code>src</code>. 
 * <p>
 * See also  {@link JavaComponentUtil#fixJavaComponentName(String)}
 * </dd> 
 * 
 * <dt>java.privateReferences</dt>
 * <dd>
 * Points to other java components whose types will be shared with this one (and maybe others)
 * Nothing referenced as private reference will be automatically exposed to the public interface of the referencing component nor to other components.
 * Anything needed to compile the private types of a Java component, must be referenced as a public reference, be part of the public types of that component, or be referenced
 * via this reference property. 
 * <br/>
 * In other words: The private types automatically see the public types and transitively anything referenced publicly as described above. In addition, 
 * to use more types in the "private implementation section" of a Java component, types that will not be exposed to referencing components, use this reference property. 
 * 
 * <p>
 * Components may be specified as a comma-separated list. Component names that have no "/" will be defaulted by appending "/java".
 * <p>
 * Alternatively this reference points to a <code>com.zfabrik.files</code> component that must have a bin folder that will be included into this java component's java resources.
 * The files resource may also have a <code>src</code> folder that will be copied befor compilation into the public interface in <code>src.impl</code> or <code>private/src</code>. 
 * <p>
 * See also {@link JavaComponentUtil#fixJavaComponentName(String)}
 * </dd> 
 * 
 * <dt>java.testReferences</dt>
 * <dd>
 * When running in development mode, the test references add to the private references of the component. Test references are meant to 
 * satisfy the dependencies of types and other definitions defined in <code>src.test</code>, <code>bin.test</code>. When not running in development
 * mode, this setting will be ignored. 
 * </dd> 
 * 
 * <dt>java.nobuild</dt>
 * <dd>
 *  A Java component specifying <code>true</code> as the value of this component property will be ignored by the compilation mechanism (as it will be in javadoc generation).
 * </dd> 
 * <dt>java.compile.order</dt>
 * <dd>
 * The compile order must be defined in java components that also contain non-java sources, e.g. <code>scala</code>, or require special handling like apsect weaving. 
 * This property can be omitted for pure java components, otherwise one has to define all compilers in the right order - e.g: <code>scala, java</code> 
 * </dd>
 * </dl> 
 * 
 *  
 * @author hb
 *
 */

public interface IJavaComponent {
	
	/**
	 * A Java component is constructed from three different aspects, that are either publicly shared, privately used for implementation, or only available 
	 * in development mode for testing. This is reflected in references, compile order, class loading and more. This enum abstracts the part
	 * when specifically referenced.
	 */
	enum Part { PUBLIC, PRIVATE, TEST } 
	
	
	/**
	 * Type constant for Java components. 
	 * @see IComponentDescriptor#COMPONENT_TYPE
	 */
	final static String TYPE = "com.zfabrik.java";
	
	
	/**
	 * Points to another java component whose public types will be shared with this one (and maybe others).
	 * <p>
	 * Everything referenced as public reference will be visible to the public interface of the referencing component as well as to all 
	 * referencing the referencing component. In other words: References are transitive.
	 * In particular, anything required to compile the public types of a Java component must be referenced via this reference property.
	 * <p>
	 * Components may be specified as a comma-separated list. Component names that have no "/" will be defaulted by appending "/java".
	 * 
	 * @see JavaComponentUtil#fixJavaComponentName(String)
	 */
	final static String PUBREFS = "java.publicReferences";
	
	/**
	 * Points to <code>com.zfabrik.files</code> or <code>com.zfabrik.java</code> components that must have a <code>bin</code> (or alternatively a <code>bin.api</code>, for Java components) folder that will be included into this java component's public java resources.
	 * The component may also have a <code>src</code> (or alternatively <code>src.api</code>, for Java components) folder that will be copied before compilation into <code>src.api</code>.
	 */
	final static String PUBINCS = "java.publicIncludes";
	
	/**
	 * Points to another java component whose public types will be shared with this one (and maybe others)
	 * Nothing referenced as private reference will be automatically exposed to the public interface of the referencing component nor to other components.
	 * Anything needed to compile the private types of a Java component, must be referenced as a public reference, be part of the public types of that component, or be referenced
	 * via this reference property. 
	 * <br/>
	 * In other words: The private types automatically see the public types and transitively anything referenced publicly as described above. In addition, 
	 * to use more types in the "private implementation section" of a Java component, types that will not be exposed to referencing components, use this reference property. 
	 * 
	 * <p>
	 * Components may be specified as a comma-separated list. Component names that have no "/" will be defaulted by appending "/java".
	 * 
	 * @see JavaComponentUtil#fixJavaComponentName(String)
	 */
	final static String PRIREFS = "java.privateReferences";
	
	/**
	 * Points to <code>com.zfabrik.files</code> or <code>com.zfabrik.java</code> components that must have a <code>bin</code> (or alternatively a <code>bin.api</code>, for Java components) folder that will be included into this java component's private java resources.
	 * The component may also have a <code>src</code> (or alternatively <code>src.api</code>, for Java components) folder that will be copied before compilation into <code>src.impl</code>.
	 */
	final static String PRIINCS = "java.privateIncludes";
	
	/**
	 * Points to another java component whose public types will be shared with this one (and maybe others) if the execution mode, as defined by the system property (see also {@link Foundation#MODE}) 
	 * is set to "development". Test references extend the private references. In conjunction with the tests source folder this allows to add test code and corresponding dependencies
	 * that will be ignored by the runtime unless running in development mode.
	 * <p>
	 * See also {@link #PRIREFS}
	 */
	final static String TESTREFS = "java.testReferences";

	/**
	 * Points to <code>com.zfabrik.files</code> or <code>com.zfabrik.java</code> components that must have a <code>bin</code> (or alternatively a <code>bin.api</code>, for Java components) folder that will be included into this java component's test java resources.
	 * The component may also have a <code>src</code> (or alternatively <code>src.api</code>, for Java components) folder that will be copied before compilation into <code>src.test</code>.
	 */
	final static String TESTINCS = "java.testIncludes";

	/**
	 *  A Java component specifying <code>true</code> as the value of this component property will be ignored by the compilation mechanism (as it will be in javadoc generation).
	 */
	final static String NOBUILD = "java.nobuild";

	/**
	 * The compile order must be defined in java components that also contain non-java sources e.g. <code>scala</code> code or that require additional handling by e.g. aspect weaving.
	 * Z2 offers a compiler extension mechanism for that purpose. This property defines the order in which compilers will be applied.
	 * This property can be omitted for pure java components in that it defaults to <code>java</code>. Otherwise one has to define all compilers in the right order - e.g: <code>scala, java</code> 
	 */
	public final static String COMPILE_ORDER = "java.compile.order";

	/**
	 * Specific override of compile order for the API part of a Java component. Otherwise like {@link #COMPILE_ORDER}.  
	 */
	public final static String PUBLIC_COMPILE_ORDER = "java.publicCompile.order";
	
	/**
	 * Specific override of compile order for the implementation part of a Java component. Otherwise like {@link #COMPILE_ORDER}.  
	 */
	public final static String PRIVATE_COMPILE_ORDER = "java.privateCompile.order";
	
	/**
	 * Specific override of compile order for the test part of a Java component. Otherwise like {@link #COMPILE_ORDER}.  
	 */
	public final static String TEST_COMPILE_ORDER = "java.testCompile.order";

	/**
	 * Default inclusion pattern for classpath computation of the component. Only library files matching this pattern
	 * will be included into classpath computation by default. This has no effect on the inclusion of .class files. 
	 */
	final static String DEFAULT_CLASSPATH_PATTERN = "^.*(?<!-sources)\\.jar$"; 
	/**
	 * Custom inclusion pattern for classpath computation of the API section. Only library files matching the specified pattern
	 * will be included into the respective classpath computation. This has no effect on the inclusion of .class files. 
	 */
	final static String PUBLIC_CLASSPATH_PATTERN = "java.publicClassPathInclusion";
	/**
	 * Custom inclusion pattern for classpath computation of the impl section. Only library files matching the specified pattern
	 * will be included into the respective classpath computation. This has no effect on the inclusion of .class files. 
	 */
	final static String PRIVATE_CLASSPATH_PATTERN = "java.privateClassPathInclusion";
	/**
	 * Custom inclusion pattern for classpath computation of the test section. Only library files matching the specified pattern
	 * will be included into the respective classpath computation. This has no effect on the inclusion of .class files. 
	 */
	final static String TEST_CLASSPATH_PATTERN = "java.testClassPathInclusion";

	/**
	 * Retrieve the private implementation's loader of this Java component. When this call completes, the Java component will have been checked and possibly compiled, if needed.  
	 */
	JavaComponentClassLoader getPrivateLoader();

	/**
	 * Retrieve the public interface's loader of this Java component. When this call completes, the Java component will have been checked and possibly compiled, if needed.  
	 */
	JavaComponentClassLoader getPublicLoader();
	
	/**
	 * Unlike the originally provided Java component repository resources made accessible via 
	 * {@link IComponentsRepository#retrieve(String)}, due to compilation the runtime resources 
	 * are the result of one or more compilation operations. Initially 
	 * the runtime resources are set as the originally provided resources. Compilation processes may
	 * alter these resources as required. The standard Java compilation process only adds 
	 * compilation results, actual .jar files and does not modify other resources.
	 * <p>
	 * Due to changes in dependency components, compilation processes might be run more than once 
	 * per actual Java component revision and different runtime instances of z2 may access different 
	 * runtime resources while sharing one set of repository resources. 
	 * <p>
	 * This method provides access to the runtime resources root.
	 */
	File getRuntimeResources();
}
