/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.java.build;

import java.io.File;
import java.net.URLClassLoader;

import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.java.IJavaComponent;

/**
 * Java component (see {@link IJavaComponent}) compiler extension interface.
 * <p>
 * This interface is implemented by implementation of components of type <code>com.zfabrik.compiler</code>.
 * <p>
 * Supported configuration properties:
 * <dl>
 * <dt>compiler.id</dt>
 * <dd>An identifier for the compiler to be used when defining the compile order
 * of a Java component using the <code>java.compile.order</code> property on the Java component.
 * </dd>
 * </dl>
 * <p>
 * Compiler extensions operate over a build result folder (usually .class files and other binary resources)
 * that will be JARred at when all compilers have completed. Compilers will be passed a class path in the 
 * form of a class loader. When required, the actual files and folders of the classpath can be retrieved
 * by casting that class loader to {@link URLClassLoader} and invoking {@link URLClassLoader#getURLs()}.
 * <p>
 * Compilers are invoked in order of the specified <code>java.compile.order</code> (see {@link IJavaComponent#COMPILE_ORDER}). When 
 * declaring a compiler, a <code>compiler.id</code> must be specified that is used to reference
 * the compiler when specifying the compiler order.
 * <p>
 * An example compiler component declaration looks like this:
 * <pre>
 * #
 * # it's a compiler
 * #
 * com.zfabrik.component.type=com.zfabrik.java.compiler
 * #
 * # it's feature_x_compiler
 * #
 * compiler.id=feature_x_compiler
 * #
 * # it's implemented here
 * #
 * component.className=com.acme.impl.FeatureXCompiler
 * </pre>
 * 
 * 
 * 
 * @author hb
 *
 */
public interface ICompiler {

	/**
	 * Compilers must define this type
	 */
	public final static String TYPE = "com.zfabrik.compiler";
	
	/**
	 * The compiler ID must be defined in the compiler component - e.g. <code>java</code> or <code>scala</code>
	 */
	public final static String COMPILER_ID = "compiler.id";
	
	/**
	 * Retrieve the "all" compiler, that uses the compile order to decide on what more specific compilers to use 
	 * @return
	 */
	default ICompiler ALL() {
		try {
			ICompiler a = IComponentsLookup.INSTANCE.lookup("com.zfabrik.boot.components.basic/allCompiler", ICompiler.class);
			if (a==null) {
				throw new IllegalStateException("allCompiler not found!");
			}
			return a; 
		} catch (Exception e) {
			RuntimeException t = new IllegalStateException("failed to load Component 'allCompiler'!",e);
			t.printStackTrace();
			throw t;
		}
	}

	/**
	 * A general purpose compiler api using a class loader as class path abstraction. 
	 * Returns true if the compilation passed otherwise it returns false.
	 * 
	 * @deprecated Implement {@link #compile(ICompilationContext)} instead.
	 * 
	 * @param compName the name of the component 
	 * @param src folder containing the source files
	 * @param dest destination folder will contain compile results
	 * @param cl class loader as class path abstraction
	 * @return true, iff compilation succeeds
	 */
	@Deprecated
	default boolean compile(String compName, File[] src, File dest, ClassLoader cl) {
		return true;
	}
	
	/**
	 * A general purpose compiler api using a class loader as class path abstraction. 
	 * Returns true if the compilation passed otherwise it returns <code>false</code>.
	 * This method to be implemented for part specific handling. It defaults to invocation of {@link #compile(String, File[], File, ClassLoader)}.
	 * 
	 * @param compName the name of the component 
	 * @param src folder containing the source files
	 * @param dest destination folder will contain compile results
	 * @param cl class loader as class path abstraction
	 * @param part of the component to compile
	 * @return true, iff compilation succeeds
	 */
	default boolean compile(ICompilationContext compilationContext) {
		return compile(
			compilationContext.getComponentName(), 
			compilationContext.getSourceFolders(),
			compilationContext.getOutputFolder(),
			compilationContext.getClassPath()
		);
	}

	/**
	 * Compiler initialization
	 */
	default void init(ICompilerContext context) {};


}
