/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Set;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.util.expression.X;

/**
 * A component repository provides component meta data and resources to the z2
 * runtime. In other words: Component repositories define the system's view of
 * the world.
 * <p>
 * This interface must be implemented by a component repository in order to
 * register at
 * {@link IComponentsManager#registerRepository(int, IComponentsRepository)}.
 * <p>
 * Component repository must be registered at meaningful times, preferrably
 * early in the startup process, to make sure components will be found by others
 * when queried.
 * <p>
 * Component Repositories are prioritized and form a chain of responsibilities,
 * given the repository with the higher priority preference over other ones with
 * lower prio. In particular, a repository of higher priority is fully
 * responsible of delegating calls to lower prio repositories.
 * <p>
 * The following table lists priorities of common component repositories:
 * <table border="1" cellspacing="0" cellpadding="5">
 * <tr>
 * <th>Repository</th>
 * <th>priority</th>
 * </tr>
 * <tr>
 * <td>dev</td>
 * <td>750</td>
 * </tr>
 * <tr>
 * <td>l1</td>
 * <td>500</td>
 * </tr>
 * </table>
 * <p>
 * Note that resources from repositories may be provided on-demand. Clients must
 * call the {@link #retrieve(String)} method to make sure component resources
 * are available locally as files.
 * <p>
 * Component descriptors however should be made available without possibly
 * unrequired resource downloads, as the system will use the descriptors to
 * implement component queries (see {@link IComponentsManager#findComponents(X)}
 * ).
 * <p>
 * Typically component repositories are structured in a two-level hierarchy
 * comprising of a <em>project name</em>, i.e. some human-managable taxonomy
 * uniquely identifying and separating namespaces, like
 * <code>com.zfabrik.aproject</code> or <code>org.apache.log4j</code>, and a
 * <em>component name</em> that separates technical components within the
 * project.
 * <p>
 * For example, a project holding a web application will typically also hold a
 * Java component and maybe a data source definition. In that case, the
 * structure within the repository may look like this (not showing other sibling
 * projects):
 * 
 * <pre>
 * myproject/
 * 	datasource.properties
 * 	java/
 * 		z.properties
 * 		src.api/...
 * 		src.impl/...
 * 	web/
 * 		z.properties
 * 		WebContent/
 * 			WEB-INF/
 * 				web.xml
 * 			index.jsp
 * </pre>
 * 
 * In this case, components may be declared as simple property sets, e.g.
 * <code>myproject/datasource</code> or as more complex components featuring a
 * folder structure like <code>myproject/web</code>. This is a commonly
 * implemented structure. Component repository implementations may however chose
 * different storage structures for component properties and component
 * hierarchies.
 * 
 * 
 * @author hb
 * 
 */
public interface IComponentsRepository {
	/**
	 * component repository implementations should add this property to
	 * component properties when providing a component descriptor so that the
	 * runtime can declare dependeny on the repository implementation component
	 * when providing a component resource.
	 */
	String COMPONENT_REPO_IMPLEMENTATION = "com.zfabrik.repo.component";
	
	/**
	 * System property indicating the operational mode the component repositories are used with.
	 * Supported values include
	 * <dl>
	 * <dt>RELAXED</dt>
	 * <dd>When set (case-independent), failure to remote connect will not stop repository operations but simply fail to discover updates. Attempts to access resources that have not been provided locally previously will still lead to errors</dd>
	 * <dt>STRICT</dt>
	 * <dd>When set (case-independent), failure to remote connect for update checks (if so required) will lead to exceptions thrown during repository operations and may render the system inoperational. This is the default setting and recommended for productive systems to ensure consistency.</dd>
	 * </dl> 
	 */
	String COMPONENT_REPO_MODE = "com.zfabrik.repo.mode";

	/**
	 * See {@link #COMPONENT_REPO_MODE}
	 */
	String COMPONENT_REPO_MODE_RELAXED = "relaxed";
	
	/**
	 * See {@link #COMPONENT_REPO_MODE}
	 */
	String COMPONENT_REPO_MODE_STRICT = "strict";
	
	/**
	 * return the most current revision of the component as available by the
	 * provider. providers must be able to count revs. A rev &lt;0 indicates
	 * that the package is not known to the repository.
	 * <p>
	 * If localOnly is set to true, only this repository is looked up. Otherwise
	 * all repositories in the chain with lower priority are considered too.
	 */
	long getRevision(String componentName, boolean localOnly) throws IOException;

	/**
	 * Shorthand for {@link #getRevision(String, boolean)} with a
	 * <code>false</code> second parameter.
	 */
	default long getRevision(String componentName) throws IOException {
		return getRevision(componentName, false);
	}

	/**
	 * finds all components satisfying a query condition. If localOnly is set to
	 * true, only this repository is queried. Otherwise all repositories in the
	 * chain with lower priority are considered too.
	 * <p>
	 * If localOnly is set to true, only this repository is looked up. Otherwise
	 * all repositories in the chain with lower priority are considered too.
	 */
	Collection<String> findComponents(X propertyExpression, boolean localOnly) throws IOException;

	/**
	 * shorthand for {@link #findComponents(X, boolean)} with a
	 * <code>false</code> second parameter.
	 */
	default Collection<String> findComponents(X propertyExpression) throws IOException {
		return findComponents(propertyExpression, false);
	}

	/**
	 * retrieves a component descriptor for a fully qualified component, e.g.
	 * <code>&lt;module&gt;/java</code> for a Java component. If localOnly is
	 * set to true, only this repository is looked up. Otherwise all
	 * repositories in the chain with lower priority are considered too.
	 * <p>
	 * If localOnly is set to true, only this repository is looked up. Otherwise
	 * all repositories in the chain with lower priority are considered too.
	 */
	IComponentDescriptor getComponent(String component, boolean localOnly);

	/**
	 * shorthand for {@link #getComponent(String, boolean)} with a
	 * <code>false</code> second parameter.
	 */
	default IComponentDescriptor getComponent(String component) {
		return getComponent(component, false);
	}

	/**
	 * Retrieve a component's resource folder. Every component has by definition
	 * a resource folder that reflects its repository defined content if any, or
	 * may be empty. This folder is a temporary structure, valid until the next
	 * time a component update will be fetched. Assuming these constraints this
	 * folder may be used to store temporary component data (e.g. as cache
	 * content).
	 * <p>
	 * A call to this method always returns a folder. Depending on the component
	 * type however, this folder may be empty, contain a single z.properties
	 * file or more.
	 * </p>
	 * <p>
	 * If localOnly is set to true, only this repository is looked up. Otherwise
	 * all repositories in the chain with lower priority are considered too.
	 * 
	 * @return the component's folder resource folder or <code>null</code>, if
	 *         the component does not exist.
	 */
	File retrieve(String name, boolean localOnly) throws IOException;

	/**
	 * shorthand for {@link #retrieve(String, boolean)} with a
	 * <code>false</code> second parameter.
	 */
	default File retrieve(String name) throws IOException {
		return retrieve(name, false);				
	}

	/**
	 * Retrieve the set of modules provided by this repository. If
	 * <code>localOnly</code> is true, the methods returns only those modules
	 * that are provided by the very repository.
	 * 
	 * @throws IOException
	 */
	Set<String> getModules(boolean localOnly) throws IOException;

	/**
	 * Retrieve the set of modules provided by this repository
	 * 
	 * @throws IOException
	 */
	default Set<String> getModules() throws IOException {
		return getModules(false);
	}

}
