/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.home.console;

import java.lang.management.ManagementFactory;
import java.util.Optional;
import java.util.logging.LogManager;
import java.util.logging.Logger;

import javax.management.MBeanServer;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import com.zfabrik.sync.SynchronizationRunner;

/**
 * Home's most fundamental MBean. This MBean offers access to synchronization 
 * and the home log.
 */
public class HomeMXBeanImpl implements HomeMXBean, NotificationListener {

	private static HomeMXBeanImpl registeredInstance;
	
	@Override
	public void handleNotification(Notification notification, Object handback) {
		System.err.println(notification);
	}
	
	private LogBuffer logBuffer = new LogBuffer();	
	
	// this runnable registers the log buffer as log handler and repeats so
	// whenever logging configuration is read again
	private Runnable logBufferRegisterer = ()->{
		Logger.getLogger("").addHandler(this.logBuffer);
	};

	public HomeMXBeanImpl() {
		// register the log buffer
		logBufferRegisterer.run();
		// and make sure it is re-registered
		LogManager.getLogManager().addConfigurationListener(logBufferRegisterer);
	}
	
	/**
	 * Clean up after unregistration
	 */
	private void destroy() {
		// unregister re-registration for log buffer
		LogManager.getLogManager().removeConfigurationListener(logBufferRegisterer);
		Logger.getLogger("").removeHandler(this.logBuffer);
	}


	
	public void synchronize() throws Exception {
		_sync(true,true);
	}

	public void verify() throws Exception {
		_sync(false,true);
	}

	public void averify() throws Exception {
		_sync(false,false);
	}

	public void asynchronize() throws Exception {
		_sync(true,false);
	}

	private void _sync(boolean invalidation, boolean synchronously) {
		final SynchronizationRunner sr = new SynchronizationRunner();
		sr.setMode((invalidation? SynchronizationRunner.INVALIDATE_AND_VERIFY:SynchronizationRunner.VERIFY_ONLY));
		sr.setKeepLog(true);
		sr.execute(synchronously);
	}
	
	public static synchronized void register() throws Exception {
		ManagementFactory.getPlatformMBeanServer().registerMBean(
			registeredInstance=new HomeMXBeanImpl(), 
			ObjectName.getInstance(HomeMXBean.OBJECTNAME)
		);
	}

	public static synchronized void unregister() throws Exception {
		MBeanServer mbs = ManagementFactory.getPlatformMBeanServer(); 
		ObjectName on = ObjectName.getInstance(HomeMXBean.OBJECTNAME);
		try {
			if (mbs.isRegistered(on)) {
				mbs.unregisterMBean(on);
			}
		} finally {
			if (registeredInstance!=null) {
				try {
					registeredInstance.destroy();
				} finally {
					registeredInstance = null;
				}
			}
		}
	}
	
	@Override
	public Long getLastSynchronizationStartTime() {
		return SynchronizationRunner.getLastSyncStarted();
	}
	
	@Override
	public Long getLastSynchronizationCompletionTime() {
		return SynchronizationRunner.getLastSyncCompleted();
	}
	
	private LogBuffer getLogBuffer() {
		return this.logBuffer;
	}
	
	@Override
	public void clear() {
		Optional.ofNullable(getLogBuffer()).ifPresent(LogBuffer::reset);
	}
	
	@Override
	public LogBufferInfo getLogBufferInfo() {
		return Optional.ofNullable(getLogBuffer()).map(LogBuffer::getInfo).orElse(null);
	}
	
	@Override
	public void configureLogBuffer(int size, long expiration, boolean extend) {
		LogBuffer logBuffer = getLogBuffer();
		if (logBuffer!=null) {
			LogBufferInfo info = logBuffer.getInfo();
			logBuffer.setMaxSize(extend? Math.max(size,info.getMaxSize()):size);
			logBuffer.setExpiration(extend? Math.max(expiration, info.getExpiration()):expiration);
		}
	}
	
	
	@Override
	public LogExcerpt getLogBufferLines(long start, int count) {
		LogBuffer logBuffer = getLogBuffer();
		if (logBuffer!=null) {
			LogExcerpt le = logBuffer.get(start, count);
			return le;
		} else {
			return null;
		}
	}

}
