/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.home.console.gui;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Image;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;

import com.zfabrik.launch.HomeLauncher;
import com.zfabrik.launch.impl.ProcessRunnerImpl;
import com.zfabrik.sync.SynchronizationRunner;
import com.zfabrik.util.internal.WorkerVault;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.util.threading.TimerUtil;

/**
 * visual console.
 *
 * @author hb
 *
 */
public class ManagementConsole {
	private final static Logger LOG = Logger.getLogger(ManagementConsole.class.getName());

	public static class ConsoleRunner implements Runnable, ActionListener, WindowListener {
		private final static Dimension DEFAULT_DIMENSION = new Dimension(700, 500);
		private static final String AC_EXIT = "EX";
		private static final String AC_SYNC = "SY";
		private static final String AC_CLEAR = "CL";
		private static final String AC_COMPL = "CO";
		private static final String AC_OFFLINE = "OF";
		private static final String AC_GO_WIKI = "WIKI";
		private static final String AC_ABOUT   = "ABOUT";
		private static final String AC_ZOOM_IN = "ZI";
		private static final String AC_ZOOM_OUT = "ZO";

		private static final int STATUS_STARTING = 0;
		private static final int STATUS_UP = 1;
		private static final int STATUS_ERRORED = 2;
		private static final int STATUS_SHUTTING_DOWN = 3;

		private boolean zombie = false;
		@SuppressWarnings("unused")
		private TextAreaLogHandler logs;
		private JPanel content;
		private JMenuBar menuBar;
		private JFrame frame;
		private JLabel statusBar;
		private JTabbedPane tabs;
		private WorkersPane workers;
		private Timer ticker;
		public  boolean refreshWorkers;
		private Updates updates;
		private JButton syncButton;
		private JCheckBox offline;
		private ImageIcon normalSyncIcon, animatedSyncIcon;

		/**
		 * GUI Scaling support
		 */
		public float scale = Float.parseFloat(System.getProperty("com.zfabrik.home.gui.scale", "1.0").trim());

		private int status = STATUS_STARTING;

		private JToolBar toolbar;

		private final class Updates implements Runnable {
			private boolean animating = false;

			private void setAnimating(boolean a) {
				if (a && !animating) {
					animating = true;
					syncButton.setIcon(animatedSyncIcon);
				} else
				if (!a && animating) {
					animating = false;
					syncButton.setIcon(normalSyncIcon);
				}
			}

			public void run() {
				offline.setSelected(Foundation.isOfflineMode());
				if (refreshWorkers) {
					workers.refresh();
				}
				switch (status) {
				case STATUS_STARTING:
					setStatusText("Starting...");
					setAnimating(true);
					break;
				case STATUS_ERRORED:
					setStatusText("Home process initialization failed. Please exit.");
					setAnimating(false);
					break;
				case STATUS_SHUTTING_DOWN:
					setStatusText("Shutting down...");
					setAnimating(true);
					break;
				default:
					if (SynchronizationRunner.sequenceExecuting()) {
						setAnimating(true);
						setStatusText("Sync'ing...");
					} else {
						setAnimating(false);
						setStatusText(null);
					}
				}
			}
		}

		private void _choosePlainFont(String key) {
			Font f = UIManager.getFont(key);
			if (f != null) {
				UIManager.put(key, new Font(f.getFamily(), Font.PLAIN, (int) (f.getSize()*scale)));
			}
		}

		private void setFonts() {
			if (Boolean.parseBoolean(System.getProperty("com.zfabrik.home.gui.fixfonts","true"))) {
				String[] fonts = new String[]{
						"Button.font", "ToggleButton.font", "RadioButton.font", "CheckBox.font", "ColorChooser.font", "ComboBox.font", "Label.font",
						"List.font", "MenuBar.font", "MenuItem.font", "RadioButtonMenuItem.font", "CheckBoxMenuItem.font", "Menu.font", "PopupMenu.font",
						"OptionPane.font", "Panel.font", "ProgressBar.font", "ScrollPane.font", "Viewport.font", "TabbedPane.font", "Table.font",
						"TableHeader.font", "TextField.font", "PasswordField.font", "TextArea.font", "TextPane.font", "EditorPane.font", "TitledBorder.font",
						"ToolBar.font", "ToolTip.font", "Tree.font"
				};
				for (String fn : fonts) {
					// make the font more agreeable.
					_choosePlainFont(fn);
				}
			}
		}
		
		public void run() {
			this.setFonts();
			
			this.frame = new JFrame("Z2 Home");
			this.frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

			this.frame.addWindowListener(this);
			this.content = new JPanel(new BorderLayout());
			frame.add(content);
			
			// add icons
			try {
				List<Image> icons = new ArrayList<>();
				for (int s : new int[]{ 16,24,32, 48,64,96,128,256,512,1024 }) {
					String path = "/icons/app/app-"+s+".png";
					URL url = ManagementConsole.class.getResource(path);
					if (url!=null) {
						icons.add(ImageIO.read(url));
					}
				}
				this.frame.setIconImages(icons);
			} catch (Exception e) {
				LOG.log(Level.WARNING,"Problem reading application icons",e);
			}

			// menu
			makeMenuBar();
			// toolbar
			makeToolBar();
			// tabs
			this.tabs = new JTabbedPane();
			// logging tab
			makeLoggingTab();

			// resource pane
			makeWorkersTab();

			// refresh of worker list
			this.tabs.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					if (tabs.getSelectedIndex() == 1) {
						workers.refresh();
						refreshWorkers = true;
					} else {
						refreshWorkers = false;
					}
				}
			});

			this.content.add(this.tabs, BorderLayout.CENTER);

			// status bar
			JPanel sb = new JPanel();
			sb.setLayout(new BorderLayout());

			this.statusBar = new JLabel();
			sb.add(this.statusBar, BorderLayout.CENTER);
			sb.setBorder(new EmptyBorder(1, 10, 1, 10));
			this.content.add(sb, BorderLayout.SOUTH);
			setStatusText("Initializing...");
			// Display the window.
			this.frame.setSize(DEFAULT_DIMENSION);
			this.frame.setVisible(true);

			// parallel refreshes
			this.ticker = TimerUtil.createTimer("workers list refresher", true);
			this.updates = new Updates();
			this.ticker.scheduleAtFixedRate(new TimerTask() {
				public void run() {
					try {
						SwingUtilities.invokeAndWait(updates);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}, 1000, 1000);

		}

		private void makeWorkersTab() {
			this.workers = new WorkersPane(this);
			this.tabs.add("Workers", this.workers.getComponent());
		}

		private void makeLoggingTab() {
			this.logs = new TextAreaLogHandler(this);
			this.tabs.addTab("log", logs.getComponent());
		}

		private void clearLoggingTab() {
			SwingUtilities.invokeLater(()->{
				synchronized (logs) {
					logs.clear();
				}
			});
		}

		private void makeToolBar() {
			JToolBar tb = new JToolBar();
			ImageIcon ic = imageIcon("/icons/stop.png");
			JButton bu = new JButton("Exit", ic);
			bu.setToolTipText("Shutdown");
			bu.setActionCommand(AC_EXIT);
			bu.addActionListener(this);
			tb.add(bu);
			ic = imageIcon("/icons/arrow_refresh.png");
			this.normalSyncIcon = ic;
			this.animatedSyncIcon = imageIcon("/icons/indicator.gif");
			bu = new JButton("Sync", ic);
			bu.setToolTipText("Force <home> synchronization");
			bu.setActionCommand(AC_SYNC);
			bu.addActionListener(this);
			this.syncButton = bu;

			tb.add(bu);
			ic = imageIcon("/icons/page_white.png");
			bu = new JButton("Clear", ic);
			bu.setActionCommand(AC_CLEAR);
			bu.addActionListener(this);
			bu.setToolTipText("Clear the logging panel");
			tb.add(bu);

			this.offline = new JCheckBox("offline");
			offline.setSelected(Foundation.isOfflineMode());
			offline.setActionCommand(AC_OFFLINE);
			offline.addActionListener(this);
			tb.add(offline);

			this.toolbar = tb;
			this.content.add(tb, BorderLayout.NORTH);
		}

		private void makeMenuBar() {
			this.menuBar = new JMenuBar();
			JMenu me;
			JMenuItem mi;
			me = new JMenu("Home");
			mi = new JMenuItem("Synchronize");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0));
			mi.addActionListener(this);
			mi.setActionCommand(AC_SYNC);
			me.add(mi);
			mi = new JMenuItem("Verify");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6, 0));
			mi.addActionListener(this);
			mi.setActionCommand(AC_COMPL);
			me.add(mi);
			mi = new JMenuItem("Clear logging panel");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F9, 0));
			mi.addActionListener(this);
			mi.setActionCommand(AC_CLEAR);
			me.add(mi);
			mi = new JMenuItem("Toggle offline");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F8, 0));
			mi.addActionListener(this);
			mi.setActionCommand(AC_OFFLINE);
			me.add(mi);
			mi = new JMenuItem("Exit");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.ALT_DOWN_MASK));
			mi.setActionCommand(AC_EXIT);
			mi.addActionListener(this);
			me.add(mi);
			this.menuBar.add(me);
			
			me = new JMenu("View");
			mi = new JMenuItem("Zoom In");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_PLUS, InputEvent.CTRL_DOWN_MASK));
			mi.addActionListener(this);
			mi.setActionCommand(AC_ZOOM_IN);
			me.add(mi);
			mi = new JMenuItem("Zoom Out");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_MINUS, InputEvent.CTRL_DOWN_MASK));
			mi.addActionListener(this);
			mi.setActionCommand(AC_ZOOM_OUT);
			me.add(mi);
			this.menuBar.add(me);

			me = new JMenu("Help");
			mi = new JMenuItem("Go to Z2 Wiki");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0));
			mi.addActionListener(this);
			mi.setActionCommand(AC_GO_WIKI);
			me.add(mi);
			mi = new JMenuItem("About...");
			mi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F11, 0));
			mi.addActionListener(this);
			mi.setActionCommand(AC_ABOUT);
			me.add(mi);
			this.menuBar.add(me);

			
			this.frame.setJMenuBar(this.menuBar);
		}
		
		public ImageIcon imageIcon(String name) {
			URL u = ManagementConsole.class.getResource(name);
			if (u == null) {
				throw new IllegalStateException("Failed to find classpath resource " + name);
			}
			ImageIcon ii = new ImageIcon(u);
			if (scale!=1.0f) {
				Image img = ii.getImage();
				img = img.getScaledInstance((int) (img.getWidth(null)*scale), (int) (img.getHeight(null)*scale), 0);
				ii = new ImageIcon(img);
			}
			return ii;
		}

		//
		// ---------------------------------------------
		//

		// This is the main operational mode actually.
		// This method is called when the home process initialized ok so far.
		// When this method ends, we shut down.
		public synchronized void waitFor() {
			if (HomeLauncher.INSTANCE.hasAutoVerify()) {
				JCheckBox cb = new JCheckBox("auto verify", true);
				cb.addItemListener(new ItemListener() {
					public void itemStateChanged(ItemEvent e) {
						boolean mode = ((JCheckBox) e.getSource()).isSelected();
						HomeLauncher.INSTANCE.setAutoVerify(mode);
						setStatusText("Toggled auto verify to " + (mode ? "on" : "off"));
					}
				});
				this.toolbar.add(cb);
			}
			try {
				if (this.status==STATUS_STARTING) {
					this.status = STATUS_UP;
					this.workers.refresh();
					setStatusText("Started");
					this.wait();
				}
				if (this.status==STATUS_ERRORED) {
					this.wait();
				}
			} catch (Throwable e) {
				e.printStackTrace();
			} finally {
				this.status=STATUS_SHUTTING_DOWN;
			}
		}

		// called when the home process failed to init.
		public synchronized void severe(final Exception e) {
			try {
				this.status = STATUS_ERRORED;
				this.setZombie();
				SwingUtilities.invokeAndWait(new Runnable() {
					public void run() {
						JOptionPane.showMessageDialog(frame, "Failed to start home process:\n" + e.toString(), "Home Startup Error", JOptionPane.ERROR_MESSAGE);
					}
				});
			} catch (Exception e1) {
				e1.printStackTrace();
			}
		}


		public void dispose() {
			JFrame f;
			Timer  t;
			synchronized (this) {
				f = this.frame;
				t = this.ticker;
				this.ticker=null;
				this.frame=null;
			}
			try {
				if (t != null) {
					t.cancel();
				}
			} finally {
				if (f != null) {
					try {
						f.setVisible(false);
					} finally {
						f.dispose();
					}
				}
			}
		}

		public void setZombie() {
			this.zombie = true;
		}

		public boolean checkZombie() {
			if (this.zombie) {
				info("The home process did not reach a functional state. Please check the console log as needed and exit this application.");
			}
			return this.zombie;
		}


		public void leave() {
			synchronized (this) {
				if (this.status!=STATUS_SHUTTING_DOWN) {
					this.status = STATUS_SHUTTING_DOWN;
					// we have a special handling for currently starting workers:
					// kill them	 all, allow no new processes.
					WorkerVault.INSTANCE.shutDown();
					// the deal is that whoever terminates us waits on us.
					this.notifyAll();
				}
			}
		}

		
		public void actionPerformed(ActionEvent e) {
			String c = e.getActionCommand();
			if (AC_ZOOM_IN.equals(c)) {
				switch (tabs.getSelectedIndex()) {
				case 0: 
					this.logs.zoomIn();
					break;
				case 1: 
					this.workers.zoomIn();
					break;
				}
			} else 
			if (AC_ZOOM_OUT.equals(c)) {
				switch (tabs.getSelectedIndex()) {
				case 0: 
					this.logs.zoomOut();
					break;
				case 1: 
					this.workers.zoomOut();
					break;
				}
			} else
			if (AC_EXIT.equals(c)) {
				setStatusText("Exit pressed");
				this.leave();
			} else 
			if (AC_CLEAR.equals(c)) {
				if (!checkZombie()) {
					clearLoggingTab();
				}
			} else 
			if (AC_SYNC.equals(c)) {
				if (!checkZombie()) {
					syncVerify(true, true);
				}
			} else 
			if (AC_COMPL.equals(c)) {
				if (!checkZombie()) {
					syncVerify(true, false);
				}
			} else 
			if (AC_OFFLINE.equals(c)) {
				if (!checkZombie()) {
					toggleOffline();
				}
			} else 
			if (AC_GO_WIKI.equals(c)) {
				if (hasDesktop()) {
					if (!new Java6DesktopSupport().goToWiki()) {
						info("Could not open browser. Possibly your Desktop is not configured");
					}
				}
			} else 
			if (AC_ABOUT.equals(c)) {
				showAbout();
			}
		}

		private void showAbout() {
			final JEditorPane p = new JEditorPane();
			p.putClientProperty(JEditorPane.HONOR_DISPLAY_PROPERTIES, Boolean.TRUE);
		    p.setFont(new Font("Arial", Font.PLAIN, 13));
			p.setEditable(false);
			p.setPreferredSize(new Dimension(280, 330));
			p.setContentType("text/html");

			// remove border and background
			p.setOpaque(false);
		    p.setBorder(BorderFactory.createEmptyBorder());

			String u = "#";
			try {
				u = this.getClass().getResource("/icons/z2.png").toURI().toString();
			} catch (Exception e) {
				e.printStackTrace();
			}
			p.setText(String.format(
					"<html><center>" +
				    "<img src=\"%s\"/>"+
					"<p style=\"font-size:15pt\">z2-Environment</p>" +
					"<br/>" +
					ProcessRunnerImpl.COPYRIGHT +
					"<br/>" +
					"<br/>" +
					"<br/>" +
					"<table>" +
					"<tr>"	+
					"<td>Core build no.:</td>" +
					"<td>%d</td>" +
					"</tr>" +
					"<tr>" +
					"<td>System id:</td>" +
					"<td>%s</td>" +
					"</tr>" +
					"</table>" +
					"<br/>" +
					"<br/>" +
					"Thank you for using Z2!" +
					"<br/>" +
					"<br/>" +
					"Wiki: <a href=\"http://redmine.z2-environment.net\">http://redmine.z2-environment.net</a><br/>"	+
					"Contact: <a href=\"mailto:contact@zfabrik.de\">contact@zfabrik.de</a>" +
					"</center></html>",
					u,
					Foundation.getCoreBuildVersion(),
					Foundation.getProperties().getProperty(Foundation.HOME_CLUSTER, "<unknown>")
					));

			p.addHierarchyListener(new HierarchyListener() {
				public void hierarchyChanged(HierarchyEvent e) {
					Window window = SwingUtilities.getWindowAncestor(p);
					if (window instanceof Dialog) {
						Dialog dialog = (Dialog) window;
						if (!dialog.isResizable()) {
							dialog.setResizable(true);
						}
					}
				}
			});

			p.addHyperlinkListener(new HyperlinkListener() {
				public void hyperlinkUpdate(final HyperlinkEvent e) {
					if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
						if (hasDesktop()) {
							try {
								new Java6DesktopSupport().goToURI(e.getURL().toURI());
							} catch (URISyntaxException e1) {
								e1.printStackTrace();
							}
						}
					}
				}
			});

			JScrollPane j = new JScrollPane(p);
			// remove border and background
			j.setBorder(BorderFactory.createEmptyBorder());
			j.setOpaque(false);
			JOptionPane.showMessageDialog(null, j, "About", JOptionPane.PLAIN_MESSAGE);
		}

		public boolean hasDesktop() {
			try {
				Class.forName("java.awt.Desktop");
				return true;
			} catch (ClassNotFoundException cnfe) {
				return false;
			}
		}

		private void syncVerify(final boolean gui, final boolean invalidate) {
			synchronized (this) {
				if (SynchronizationRunner.sequenceExecuting()) {
					if (gui) {
						info("A synchronization/verification is currently running. Please wait until it has completed.");
					}
				} else {
					setStatusText("Sync/Verify...");
					SynchronizationRunner runner = new SynchronizationRunner((invalidate ? SynchronizationRunner.INVALIDATE_AND_VERIFY
							: SynchronizationRunner.VERIFY_ONLY));
					runner.setWhenDone(new Runnable() {
						public void run() {
							SwingUtilities.invokeLater(new Runnable() {
								public void run() {
									workers.refresh();
								}
							});
						}
					});
					runner.execute(false);
				}
			}
		}


		private void toggleOffline() {
			System.setProperty(
				Foundation.OFFLINE,
				Boolean.toString(!Foundation.isOfflineMode())
			);
			offline.setSelected(Foundation.isOfflineMode());
			LOG.info(Foundation.isOfflineMode()? "Going offline":"Going online");
		}


		private void info(String string) {
			JOptionPane.showMessageDialog(this.frame, string, "Information", JOptionPane.INFORMATION_MESSAGE);
		}

		private final static DateFormat df = DateFormat.getTimeInstance(DateFormat.MEDIUM);
		private String statusText;

		private void setStatusText(String s) {
			if (s!=null) {
				if (!s.equals(statusText)) {
					this.statusBar.setText(new StringBuilder(50).append(df.format(new Date())).append(": ").append(s).toString());
					statusText = s;
				}
			} else {
				if (statusText!=null) {
					statusText=s;
					this.statusBar.setText(" ");
				}
			}
		}

		public void windowActivated(WindowEvent e) {
		}

		public synchronized void windowClosed(WindowEvent e) {
			this.leave(); // leaves
		};

		public void windowClosing(WindowEvent e) {
			this.leave(); // leaves
		}

		public void windowDeactivated(WindowEvent e) {
		}

		public void windowDeiconified(WindowEvent e) {
		}

		public void windowIconified(WindowEvent e) {
		}

		public void windowOpened(WindowEvent e) {
		}

		public float getScale() {
			// TODO Auto-generated method stub
			return this.scale;
		}


	}

	private ConsoleRunner app = new ConsoleRunner();

	public void init() throws Exception {
		SwingUtilities.invokeAndWait(app);
	}

	public void waitFor() {
		this.app.waitFor();
	}

	public void verify() throws Exception {
		SwingUtilities.invokeAndWait(new Runnable() {
			public void run() {
				app.syncVerify(false, false);
			}
		});
	}

	public void severe(final Exception e) {
		this.app.severe(e);
		this.app.waitFor();
	}

	public void leave() {
		this.app.leave();
	}

	public void dispose() {
		try {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					app.dispose();
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void setTitle(final String title) {
		try {
			SwingUtilities.invokeAndWait(new Runnable() {
				public void run() {
					app.frame.setTitle(title);
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public Icon imageIcon(String name) {
		return app.imageIcon(name);
	}

	
	
}