/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.home.console.gui;

import java.awt.Component;
import java.awt.Font;
import java.awt.event.MouseWheelEvent;
import java.util.logging.Handler;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.text.BadLocationException;

import com.zfabrik.home.console.gui.ManagementConsole.ConsoleRunner;
import com.zfabrik.util.logging.CompactFormatter;

public class TextAreaLogHandler extends Handler {
	
	// check twice a second whether the log handler is still registered
	private final static long LOGHANDLER_CHECK_DELAY = 500; 
	
	// we keep may 2000 lines
	private final static int MAX_LINES = 2000;
	// and max 1000 chars per line
	private final static int MAX_LINE_LENGTH = 1000;
	// and max MAX_LINES*MAX_LINE_LENGTH+1 buffer
	private final static int MAX_BUFFER_SIZE = (MAX_LINE_LENGTH+1)*MAX_LINES;
	
	private StringBuilder toAppend = new StringBuilder(MAX_LINE_LENGTH);

	private Runnable appender = new Runnable() {
		public void run() {
			checkLogHandlerAssignment();
			synchronized (appender) {
				if (toAppend.length() > 0) {
					textArea.append(toAppend.toString());
					// release the buffer
					toAppend = new StringBuilder(MAX_LINE_LENGTH);
					int l = textArea.getLineCount();
					if (l > MAX_LINES) {
						l -= MAX_LINES;
						try {
							int o = textArea.getLineStartOffset(l);
							textArea.getDocument().remove(0, o);
						} catch (BadLocationException e) {
							e.printStackTrace();
						}
					}
				}
			}
		}
	};
	
	public void publish(LogRecord record) {
		String line = fm.format(record);
		if (line.length()>MAX_LINE_LENGTH) {
			line = line.substring(0,MAX_LINE_LENGTH-3)+"...";
		}
		synchronized (appender) {
			if (toAppend.length()<MAX_BUFFER_SIZE) {
				toAppend.append(line);
				if (toAppend.length()>=MAX_BUFFER_SIZE) {
					// stop appending
					toAppend.append("(log buffer limit reached - will stop appending)\n");
				}
				SwingUtilities.invokeLater(appender);
			}
		}
	}
	
	//
	// make sure our logging handler is registered.
	// logging reconfig resets the handler
	//
	private synchronized void checkLogHandlerAssignment() {
		Logger l = Logger.getLogger("");
		Handler[] hs = l.getHandlers();
		for (Handler h:hs) {
			if (h==this) {
				return;
			}
		}
		l.addHandler(this);
	}


	private CompactFormatter fm = new CompactFormatter();
	private JTextArea textArea;
	private JScrollPane pane;

	public TextAreaLogHandler(ConsoleRunner runner) {
		this.textArea = new JTextArea();
		this.textArea.setEditable(false);
		Font font = new Font(Font.MONOSPACED, Font.PLAIN, (int) (12* runner.getScale()));
		this.textArea.setFont(font);
		
		this.pane = new JScrollPane(this.textArea);
		this.textArea.addMouseWheelListener((MouseWheelEvent e) -> {
			if (e.isControlDown()) {
				int notches = e.getWheelRotation();
				if (notches>0) {
					zoomOut();
				}
				if (notches<0) {
					zoomIn();
				}
			} else {
				// propagate to pane
				pane.dispatchEvent(e);
			}
		});
		
		checkLogHandlerAssignment();
		// force recurring log handler checking
		Thread t = new Thread(new Runnable() {
			public void run() {
				while (true) {
					checkLogHandlerAssignment();
					try {
						Thread.sleep(LOGHANDLER_CHECK_DELAY);
					} catch (InterruptedException ie) {
						break; // terminate
					}
				}
			}
		});
		t.setDaemon(true);
		t.start();
	}

	public void flush() {
	}

	public void close() throws SecurityException {
	}
	
	public void zoomIn() {
		if (textArea.getFont().getSize()<36) { 
			textArea.setFont(textArea.getFont().deriveFont((float)(textArea.getFont().getSize()+1)));
			textArea.repaint();
			textArea.revalidate();
		}
	}
	
	public void zoomOut() {
		if (textArea.getFont().getSize()>1) { 
			textArea.setFont(textArea.getFont().deriveFont((float)(textArea.getFont().getSize()-1)));
			textArea.repaint();
			textArea.revalidate();
		}
	}
	
	public Component getComponent() {
		return pane;
	}

	public void clear() {
		try{
			this.textArea.getDocument().remove(0,this.textArea.getDocument().getLength());
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
	}
}
