/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components;

import static com.zfabrik.components.java.JavaComponentUtil.privateContextExceptionExecute;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.concurrent.Callable;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IDependencyComponent;
import com.zfabrik.components.java.JavaComponentUtil;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.util.threading.ThreadUtil;

/**
 * A simple shared util to eagerly and dynamically load 
 * component or state dependencies.
 * 
 * @author hb
 *
 */
public class ComponentDependencyUtil {
	
	
	public static interface IDependencyResolver {
		Collection<String> getDependencies() throws Exception;
	}
	
	
	public static void prepareDependencies(IResourceHandle dependant, IDependencyResolver resolver, Logger logger) {
		Set<String> depsDone = new HashSet<String>();
		Set<String> depsRemain = new HashSet<String>();
		String name = dependant.getResourceInfo().getName();
		boolean failed = false;
		do {
			// get all deps
			try {
				depsRemain.addAll(resolver.getDependencies());
			} catch (Exception e) {
				throw new IllegalStateException("Failed to collect component dependencies: "+name,e);
			}

			depsRemain.removeAll(depsDone);
			if (!depsRemain.isEmpty()) {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("Component has remaining dependencies ("+depsRemain +"): "+name);
				}
				IResourceHandle rh;
				// greedily try to load
				for (String cn : depsRemain) {
					rh = IComponentsLookup.INSTANCE.lookup(cn, IResourceHandle.class);
					dependant.addDependency(rh);
					try {
						final IDependencyComponent r = rh.as(IDependencyComponent.class);
						if (r!=null) {
							privateContextExceptionExecute(
								cn,
								new Callable<Void>() {
									public Void call() throws Exception {
										r.prepare();
										return null;
									}
								}
							);
						}
					} catch (Exception e) {
						failed = true;
						logger.log(Level.SEVERE, "Error while preparing dependency (" + cn + "): " + name, e);					
					} finally {
						depsDone.add(cn);
					}
				}
			}
		} while (!depsRemain.isEmpty());
		if (failed) {
			throw new IllegalStateException("Failed to prepare dependencies (see logged exceptions): "+name);
		}
	}
	
	public static void addDepsFromCSL(Set<String> deps, String dep) {
		if (dep!=null) {
			StringTokenizer tk = new StringTokenizer(dep,",");
			while (tk.hasMoreTokens()) {
				String d = tk.nextToken().trim();
				if (d.length()>0) {
					deps.add(d);
				}
			}
		}
	}


}
