/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.resources;

import java.util.Set;
import java.util.logging.Logger;

import com.zfabrik.impl.resources.NamespaceImpl.LookupType;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.IResourceInfo;
import com.zfabrik.resources.IResourceObserver;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.TypeRef;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.work.CycleProtector;

public class ResourceHandleImpl implements IResourceHandle {
	
	/**
	 * The resource handle holds on the dependend resources directly, so that dependencies may not be collected before 
	 * the dependent resource. This member is kept here for the above reason, but it is updated from {@link ResourceEnv}. 
	 */
	protected Set<ResourceHandleImpl> deps;
	protected Resource resource;
	protected ResourceEnv env;
	private boolean inAs; 
	
	public IResourceInfo getResourceInfo() {
		return this.env.ns.getResourceInfo(env.vh);
	}
	
	protected synchronized ResourceHandleImpl toOutside() {
		if (this.resource==null) {
			return null;
		} else {
			return this;
		}
	}

	protected boolean _isAssigned() {
		return this.resource!=null;
	}
	
	protected void _assign(Resource resource) {
		this.resource=resource;
		this.resource.init(this);
	}
		
	public ResourceHandleImpl() {}

	/**
	 * Delegation facade to {@link Resource#as(Class)}. 
	 */
	protected <T> T as(TypeRef<T> typeRef, LookupType lookupType) {
		return CycleProtector.supply("as:"+this.env.getName()+":"+typeRef.getType(), ()->{
			if (IResourceHandle.class.equals(typeRef.getType())) {
				return typeRef.getClz().cast(this);
			} else 
			if (IResourceObserver.class.equals(typeRef.getType())) {
				return typeRef.getClz().cast(this.env);
			} else {
				Resource r;
				try {
					synchronized (this) {
						this.inAs=true; 
						r = this.resource;
						if (r==null)
							throw new IllegalStateException("Invalid resource handle (not assigned)");
					}
					// DO NOT synchronize around the actual resource call. This can easily lead to deadlocks, as resuource call RM and vice versa!
					// There is a problem with weak dependencies whose invalidation call may happen 
					// right when the resource set up new dependencies. Bad luck!
					// (after all, we had a 12h downtime on Aug. 04, 2009 due to a deadlock exactly here!!!
					//
					// #2047: We always go by type ref. The default implementation of Resource tries to delegate to as(Class).
					// Overwriting as(TypeRef) in Resource would get ALL as-calls this way. While all class based calls would
					// end up in as(Class) normally.
					return r.as(typeRef);
				} finally {
					synchronized (this) {
						this.inAs=false; 
					}				
				}
			}
		});
	}
	
	public <T> T as(Class<T> clz) {
		return as(new SimpleTypeRef<T>(clz), LookupType.GET_BY_CLASS);
	}
	
	@Override
	public <T> T as(TypeRef<T> typeRef) {
		return as(typeRef, LookupType.GET_BY_TYPE_REF);
	}

	protected void _invalidateLocal(boolean force)
			throws ResourceBusyException {
		Resource r;
		synchronized (this) {
			if (this.inAs) {
				logger.warning("Resource invalidated while in as(<...>): "+this.env.getName());
			}
			r = this.resource;
		}
		if (r!=null) {
			this.resource.invalidate();
			this.adjust(0,Long.MAX_VALUE,IResourceHandle.WEAK);
		}
	}

	public void adjust(long ttl, long exp, short rm) {
		this.env.ns.adjust(this.env.vh, ttl, exp, rm);
	}
	
	public IResourceHandle getHandle() {
		return this;
	}
	
	public ResourceEnv env() {
		return this.env;
	}

	public IResourceObserver getObserver() {
		return this.env;
	}

	public void invalidate(boolean force)
			throws ResourceBusyException {
		NamespaceImpl ni = this.env.ns;
		if (ni!=null) {
			ni._invalidate(this.env,force);
		}
	}

	public void addDependency(IResourceHandle rh) {
		ResourceHandleImpl rhi = (ResourceHandleImpl) rh;
		this.env.addDependency(rhi);
	}
	
	public void removeDependency(IResourceHandle rh) {
		ResourceHandleImpl rhi = (ResourceHandleImpl) rh;
		this.env.removeDependency(rhi.env);
	}
	
	public boolean hasDependency(IResourceHandle rh) {
		return this.env.hasDependency(((ResourceHandleImpl)rh).env);
	}
	
	public void attach(Object o) {
		this.env.ns.attach(o, this);
	}

	public void detach(Object o) {
		this.env.ns.detach(o);
	}

	public String toString() {
		return this.env.getName();
	}
	
	private final static Logger logger = ResourceManagerImpl.logger;
}
