/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.resources;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.impl.resources.NamespaceImpl.LookupType;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.IResourceLookup;
import com.zfabrik.resources.IResourceManager;
import com.zfabrik.resources.TypeRef;
import com.zfabrik.resources.provider.IResourceProvider;
import com.zfabrik.resources.provider.IResourceProviderBuilder;

public class ResourceManagerImpl implements IResourceManager {
	private Set<IResourceProviderBuilder> builders = new HashSet<IResourceProviderBuilder>();
	
//	/* ------------------
//	 * In order to lock resources during unload in a transitive way, i.e. so that during an unload (or mass unload) operation invalidated resources
//	 * do not request resources that have just been unloaded, we manage a re-entrant-capable book-keeping of unloaded handles, 
//	 * that will be freed again, once the outermost call has completed.
//	 * 
//	 * This turned out to be a bad idea, as verification runs will not be consistent if they hit an otherwise completely
//	 * reasonable administrative invalidation (as e.g. in the KULT editor use-case). Hence: Resources must maintain a clear locking
//	 * strategy for their own state (e.g. on this, from dependant to dependency). 
//	 * ------------------
//	 */
//
//	private static class InvalidationLog {
//		int nesting=1;
//	}
//
//	protected static ThreadLocal<InvalidationLog> invalidationLog = new ThreadLocal<InvalidationLog>();
//	
//	protected static void il_enter() {
//		if (invalidationLog.get()==null) {
//			invalidationLog.set(new InvalidationLog());
//		} else {
//			invalidationLog.get().nesting++;
//		}
//	}
	/*
	 * Generic resolve. We first resolve the namespace and then delegate to the namespace
	 */
	private <T> T resolve(String resourceId, TypeRef<T> typeRef, LookupType lookupType) {
		String nn,ns;
		int p = resourceId.indexOf('/');
		if (p<0) {
			ns = resourceId;
			nn = null;
		} else {
			ns = resourceId.substring(0,p);
			nn = resourceId.substring(p+1);
		}
		NamespaceImpl ni = this.getNamespace(ns);
		if (nn==null) {
			if (IResourceLookup.class.equals(typeRef.getType())) {
				return typeRef.getClz().cast(ni);
			}
			throw new IllegalArgumentException("missing namespace in resource name "+resourceId);
		} else {
			if (ni!=null) {
				return ni.resolve(nn, typeRef, lookupType);
			} else {
				return null;
			}
		}
	}
	
	
	public <T> T lookup(String resourceId, TypeRef<T> typeRef) {
		return resolve(resourceId, typeRef, LookupType.GET_BY_TYPE_REF);
	}
	
	public <T> T lookup(String resourceId, Class<T> clz) {
		return resolve(resourceId, new SimpleTypeRef<T>(clz), LookupType.GET_BY_CLASS);
	}

	public <T> T query(String resourceId, TypeRef<T> typeRef) {
		return resolve(resourceId, typeRef, LookupType.QUERY_BY_TYPE_REF);
	}

	public <T> T query(String resourceId, Class<T> clz) {
		return resolve(resourceId, new SimpleTypeRef<T>(clz), LookupType.QUERY_BY_CLASS);
	}

	public IResourceLookup registerProvider(String namespace, IResourceProvider prov) {
		synchronized (this.resources) {
			if (this.resources.containsKey(namespace)) {
				throw new IllegalArgumentException("Cannot register resource provider for name space "+namespace+": namespace already in use");
			}
			NamespaceImpl nsi = new NamespaceImpl(namespace, prov);
			prov.init(new ResourceProviderContextImpl(nsi));
			// now put the name space as available
			this.resources.put(namespace, nsi);
			return nsi;
		}
	}

	public void unregisterProvider(String namespace,boolean forced) {
		if (logger.isLoggable(Level.FINE)) {
			logger.fine("unregistering resource provider: "+namespace);
		}
		try {
			NamespaceImpl ns = this.queryNamespace(namespace);
			if (ns!=null) 
				ns.release(forced);
		} finally {
			synchronized (this.resources) {
				this.resources.remove(namespace);
			}
		}
	}
	
	
	public void registerProviderBuilder(IResourceProviderBuilder builder) {
		synchronized (this.builders) {
			this.builders.add(builder);
		}
	}
	
	public void unregisterProviderBuilder(IResourceProviderBuilder builder) {
		synchronized (this.builders) {
			this.builders.remove(builder);
		}
	}
	
	public List<String> list() {
		synchronized (resources) {
			return new ArrayList<String>(this.resources.keySet());
		}
	}
	
	// the actual resources kept
	// name space ---> ( name ---> resource handle)
	private Map<String,NamespaceImpl> resources =  new HashMap<String, NamespaceImpl>();

	protected NamespaceImpl getNamespace(String namespace) {
		NamespaceImpl ns;
		synchronized (this.builders) {
			synchronized (resources) {
				ns = this.resources.get(namespace);
			}
			if (ns==null) {
				// find a builder.
				for (IResourceProviderBuilder b : this.builders) {
					if ((b.findProvider(namespace))!=null) {
						synchronized (resources) {
							ns = this.resources.get(namespace);
						}
						break;
					}
				}
			}
		}
		return ns;
	}

	private NamespaceImpl queryNamespace(String namespace) {
		synchronized (this.builders) {
			synchronized (resources) {
				return this.resources.get(namespace);
			}
		}
	}

	public int  invalidate(Collection<String> invs) {
		int i=0;
		for (String cn : invs) {
			IResourceHandle rh = query(cn, IResourceHandle.class);
			if (rh != null) {
				i++;
				try {
					rh.invalidate(true);
				} catch (Exception e) {
					logger.log(Level.WARNING, "Exception while invalidating component: " + cn, e);
				}
			}
		}
		return i;
	}	
	
	protected final static Logger logger = Logger.getLogger(ResourceManagerImpl.class.getName());

	// shutdown. We keep iterating until all name spaces are released
	private void destroy() {
		String ns;
		do {
			ns=null;
			synchronized (this.resources) {
				Iterator<String> it = this.resources.keySet().iterator();
				if (it.hasNext()) {
					ns = it.next();
				}
			}
			if (ns!=null) {
				try {
					unregisterProvider(ns,true);
				} catch (Exception e) {
					logger.log(Level.WARNING,"Exception while releasing namespace: "+ns,e);
				}
			}
		} while (ns!=null);		
	}
	
	public static void initialize() {
		// does nothing really
	}
	
	public static void shutdown() {
		((ResourceManagerImpl)INSTANCE).destroy();
	}
	
}
