package com.zfabrik.launch;

/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import com.zfabrik.launch.impl.Z2CommandLine;


public class CommandLineComputer extends Z2CommandLine {
	private static final String CONFIG_FILE = "bin/launch.properties";
	private static final String EMODE = "-Emode";
	private static final String MODE = "-mode";
	private static final String JAVAW = "-javaw";
	private final static String VMOPTS = "home.vmopts";
	private Map<String,String> 	vmopts = new HashMap<>();
	private Map<String,String> 	progopts = new HashMap<>();
	
	/**
	 * This class computes a java commandline for a home process.
	 * It does so based on the actual command line and whatever
	 * defaults are defined in the runtime configuration.
	 * At first we evaluate the runtime properties as far as they
	 * define vm properties. At last we check the command line.
	 */
	public static void main(String[] args) throws Exception {
		new CommandLineComputer(args);
	}
	
	
	public CommandLineComputer(String[] args) throws Exception {
		// start with defaults from vm options in launch.properties
		Properties rp = new Properties();
		File launchPropFile = new File(getZ2Home(),CONFIG_FILE);
		try (FileInputStream fin = new FileInputStream(launchPropFile)) {
			rp.load(fin);
			processOptsLine(rp.getProperty(VMOPTS));
		} catch (FileNotFoundException fnfe) {
			log.warning("No config file "+launchPropFile.getAbsolutePath()+" found. Ignoring.");
		}
		
		{
			// split the args into the set for the vm and for the application 
			List<String> vm = new ArrayList<>();
			List<String> ap = new ArrayList<>();
			List<String> cu = vm;
			for (String a : args) {
				if ("-".equals(a)) {
					// the separator. Switch over
					cu = ap;
				} else {
					cu.add(a);
				}
			}
			try {
				parseArgs(
					vm,
					(n,v)-> {
						String value = v.substring(n.length()+1).trim();
						// given a mode, process the whole launch.properties
						if (EMODE.equals(n)) {
							processOptsLine(rp.getProperty(VMOPTS+"."+value));
						} else
						if (MODE.equals(n)) {
							processOptsLine(rp.getProperty(VMOPTS+"."+value));
						} else {
							// otherwise add (and possibly override)
							this.vmopts.put(n,v);
						}
					}
				);
				// and finally collect the application options 
				parseArgs(ap,this.progopts::put);
			} catch (IllegalArgumentException iae) {
				log.severe(iae.getMessage());
				usage();
			}
		}
		// show help, if requested
		if (this.vmopts.containsKey(HELP)) {
			usage();
		}
		boolean useJavaw = this.vmopts.containsKey(JAVAW);

		// build the command line
		StringBuffer cmdline;
		if (useJavaw) {
			cmdline = new StringBuffer("javaw");
		} else {
			cmdline = new StringBuffer("java");
		}
		vmopts.forEach((n,v)->cmdline.append(" ").append(v));
		cmdline.append(" ").append(HomeLauncher.class.getName());
		progopts.forEach((n,v)->cmdline.append(" ").append(v));
		System.out.print(cmdline.toString());		
	}


	private void processOptsLine(String property) {
		if (property==null) return;
		parseArgs(Arrays.asList(property.split(" ")),this.vmopts::put);
	}

}
