/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.resources;

import java.util.List;

/**
 * Access to the resource management system. Implemented by {@link IResourceManager}.
 * 
 * @author hb
 *
 */
public interface IResourceLookup {

	/**
	 * Type safe lookup for resource identified by a name. If the named resource is not
	 * loaded into memory as of the moment of this call, a corresponding provider will be
	 * looked up and asked to provide the resource. Note that 
	 * nesting is only two levels deep. 
	 * On the outer level of the resource manager, names are of the form	  
	 * &lt;name space&gt;/&lt;name&gt; while name space local lookups use their
	 * own naming scheme for which all characters but question marks are allowed.
	 *  
	 *  
	 * @param resourceId a resource name name. 
	 * @param clz expected resource type.
	 * @return resource instance of <code>null</code> if the resource
	 * is not available
	 */
	<T> T lookup(String resourceId,Class<T> clz);
	
	/**
	 * Type safe lookup for resource identified by a name. This method will only return non-null if the resource
	 * in question has been loaded into memory. Note that 
	 * nesting is only two levels deep. 
	 * On the outer level of the resource manager, names are of the form	  
	 * &lt;name space&gt;/&lt;name&gt; while name space local lookups use their
	 * own naming scheme.
	 *  
	 *  
	 * @param resourceId a resource name name
	 * @param clz expected resource type.
	 * @return resource instance of <code>null</code> if the resource
	 * is not available
	 */
	<T> T query(String resourceId,Class<T> clz);

	/**
	 * Type safe lookup for resource identified by a name. If the named resource is not
	 * loaded into memory as of the moment of this call, a corresponding provider will be
	 * looked up and asked to provide the resource. Note that 
	 * nesting is only two levels deep. 
	 * On the outer level of the resource manager, names are of the form	  
	 * &lt;name space&gt;/&lt;name&gt; while name space local lookups use their
	 * own naming scheme for which all characters but question marks are allowed.
	 *  
	 * Using the {@link TypeRef} API, it is possible to specify parameterized types in a type-safe manner. For example:
	 * <pre>
	 * <code>
	 * lookup(new TypeRef&lt;List&lt;String&gt;&gt;(){});
	 * </code>
	 * </pre>
	 * 
	 *  
	 * @param resourceId a resource name name. 
	 * @param typeRef expected resource type as {@link TypeRef} supporting parameterized types.
	 * @return resource instance of <code>null</code> if the resource
	 * is not available
	 */
	<T> T lookup(String resourceId,TypeRef<T> typeRef);
	
	/**
	 * Type safe lookup for resource identified by a name. This method will only return non-null if the resource
	 * in question has been loaded into memory. Note that 
	 * nesting is only two levels deep. 
	 * On the outer level of the resource manager, names are of the form	  
	 * &lt;name space&gt;/&lt;name&gt; while name space local lookups use their
	 * own naming scheme.
	 *  
	 * Using the {@link TypeRef} API, it is possible to specify parameterized types in a type-safe manner. For example:
	 * <pre>
	 * <code>
	 * lookup(new TypeRef&lt;List&lt;String&gt;&gt;(){});
	 * </code>
	 * </pre>
	 *  
	 * @param resourceId a resource name name
	 * @param typeRef expected resource type as {@link TypeRef} supporting parameterized types.
	 * @return resource instance of <code>null</code> if the resource
	 * is not available
	 */
	<T> T query(String resourceId,TypeRef<T> typeRef);

	/**
	 * return an iterator of bound names (active resources) that is usually less than what can 
	 * be served.
	 */
	List<String> list();
}
