/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.resources;

import java.util.Collection;

import com.zfabrik.resources.provider.IResourceProvider;
import com.zfabrik.resources.provider.IResourceProviderBuilder;

/**
 * core manager of the resource management system. Lookups for single resources can be performed like this:
 * 
 * <pre>
 * IResourceManager.INSTANCE.lookup(&lt;resource name&gt;, &lt;return type&gt;);
 * </pre>
 * 
 * A namespace local lookup facility can be retrieved by a special lookup style, e.g.:
 * <pre>
 * IResourceManager.INSTANCE.lookup("com.zfabrik.components",IResourceLookup.class);
 * </pre>
 * 
 * 
 * @author hb
 *
 */
public interface IResourceManager extends IResourceLookup {
	 
	// initialization trick
	public static class initializer {
		private static IResourceManager get() {
			try {
				Class<?> clz = Class.forName("com.zfabrik.impl.resources.ResourceManagerImpl");
				return (IResourceManager) clz.getConstructor().newInstance();
			} catch (Exception e) {
				RuntimeException t = new IllegalStateException("failed to load the resource manager implementation. Exiting!",e);
				t.printStackTrace();
				throw t;
			}
		}
	}
	
	/**
	 * Use as <code>IComponentsLookup.INSTANCE</code> to get the implementation of this interface. 
	 */
	static IResourceManager INSTANCE = initializer.get();
	
	/**
	 * register a provider for resources of a name space.
	 * @param namespace
	 * @param prov
	 * @return A {@link IResourceLookup} instance for the registered provider 
	 */
	IResourceLookup registerProvider(String namespace, IResourceProvider prov);
	/**
	 * unregister a provider for resources of a name space.
	 */
	void unregisterProvider(String namespace, boolean forced);
	
	/**
	 * register a {@link IResourceProviderBuilder}
	 * @param builder
	 */
	void registerProviderBuilder(IResourceProviderBuilder builder);
	/** 
	 * unregister the provider builder if present
	 */
	void unregisterProviderBuilder(IResourceProviderBuilder builder);

	/**
	 * mass invalidation (locking resources on the way)
	 */
	int  invalidate(Collection<String> invs);
	
}
