/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.logging;

import java.lang.management.ManagementFactory;
import java.util.logging.LogRecord;
import java.util.logging.StreamHandler;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.StandardMBean;

/**
 * LogHandler that provides the JUL log outputs as a counter per log-level for JMX.
 * <p>
 * Add this handler to the JUL logging.properties configuration file to the {@code handlers} property:
 * <pre>
 *     .level=INFO
 *     ...
 *     handlers=...,com.zfabrik.util.logging.JMXLogHandler
 *     ...
 * </pre>
 *
 * The handler exposes the following counters as JMX attributes under {@code zfabrik/com.zfabrik.logging/JULMetrics}:
 * <ul>
 *     <li>LogFinest - counts the number of log entries of the FINEST level</li>
 *     <li>LogFiner - counts the number of log entries of the FINER level</li>
 *     <li>LogFine - counts the number of log entries of the FINE level</li>
 *     <li>LogConfig - counts the number of log entries of the CONFIG level</li>
 *     <li>LogInfo - counts the number of log entries of the INFO level</li>
 *     <li>LogWarning - counts the number of log entries of the WARNING level</li>
 *     <li>LogSevere - counts the number of log entries of the SEVERE level</li>
 * </ul>
 */
public class JMXLogHandler extends StreamHandler {

    /** JMX Object Name */
    private static final String     JUL_METRIC_JMX_NAME = "zfabrik:type=com.zfabrik.logging,name=JULMetrics";
    private static final ObjectName JUL_METRIC_JMX_OBJECT_NAME;

    static {
        try {
            JUL_METRIC_JMX_OBJECT_NAME = ObjectName.getInstance(JUL_METRIC_JMX_NAME);
        } catch (MalformedObjectNameException e) {
            throw new RuntimeException("Invalid Object-Name " + JUL_METRIC_JMX_NAME);
        }
    }

    private final LogLevelMBeanImpl logLevelMBean;

    public JMXLogHandler() {
        try {
            MBeanServer server = ManagementFactory.getPlatformMBeanServer();
            if (server.isRegistered(JUL_METRIC_JMX_OBJECT_NAME)) {
                // When logging.properties is changes (e.g. changing a log level) the handlers are re-created!
                // In this case the previously registered MBean instance must be unregistered.
                server.unregisterMBean(JUL_METRIC_JMX_OBJECT_NAME);
            }

            logLevelMBean = new LogLevelMBeanImpl();
            server.registerMBean(
                    new StandardMBean(logLevelMBean, LogLevelMBean.class),
                    JUL_METRIC_JMX_OBJECT_NAME
            );
        } catch (Exception e) {
            throw new RuntimeException("Failed to register JMXLogHandler MBean " + JUL_METRIC_JMX_NAME, e);
        }
    }

    @Override
    public synchronized void publish(LogRecord record) {
        logLevelMBean.incLogLevel(record);
    }
}
