/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.work;

import java.util.LinkedList;
import java.util.concurrent.Callable;
import java.util.function.Supplier;

/**
 * A generic cycle detection utility that can be used in methods that may run into 
 * recursive control flows.
 */
public class CycleProtector {
	
	/**
	 * Exception thrown when a cycle was detected
	 */
	public static class CycleDetectedException extends RuntimeException {
		private static final long serialVersionUID = 1L;
		
		public CycleDetectedException(String state, String trigger) {
			super("At "+state+" entering "+trigger);
		}

	}
	
	/**
	 * Exception thrown when a protection stack was not cleanly resolved
	 * at the end of a work unit. 
	 */
	public static class PendingCycleProtectionException extends RuntimeException {
		private static final long serialVersionUID = 1L;
		
		public PendingCycleProtectionException(String message) {
			super(message);
		}
	}

	
	private static ThreadLocal<LinkedList<String>> nodes = new ThreadLocal<>();
	
	/**
	 * Enter a control flow that is not to be traversed again deeper. This may be some dependency inspection, some graph traversal, 
	 * anything where a cyclic repetition is to be avoided. The node string passed identifies the control flow.
	 */
	public static void enter(String node) throws CycleDetectedException {
		if (nodes.get()==null) {
			nodes.set(new LinkedList<>());
		}
		if (!nodes.get().contains(node)) {
			nodes.get().add(0,node);
		} else {
			throw new CycleDetectedException(stateToString(),node);
		}
	}
	
	/**
	 * Forcibly clear the protector. Use with caution, if at all
	 */
	public static void clear() {
		nodes.set(null);
	}

	/**
	 * Leave a control flow again. This must be the last entered. 
	 */
	public static void leave(String node) throws CycleDetectedException {
		if (nodes.get()==null) {
			throw new IllegalStateException("Nothing to leave!");
		}
		if (!node.equals(nodes.get().getFirst())) {
			throw new IllegalStateException("Cannot leave "+node+" as we are in "+nodes.get().peek());
		}
		nodes.get().removeFirst();
		if (nodes.get().isEmpty()) {
			nodes.set(null);
		}
	}
	
	/**
	 * Run a control flow
	 */
	public static void run(String node, Runnable run) {
		enter(node);
		try {
			run.run();
		} finally {
			leave(node);
		}
	}

	/**
	 * Call a control flow as {@link Callable}
	 */
	public static <T> T call(String node, Callable<T> call) throws Exception {
		enter(node);
		try {
			return call.call();
		} finally {
			leave(node);
		}
	}

	/**
	 * Call a control flow as Supplier
	 */
	public static <T> T supply(String node, Supplier<T> supplier) {
		enter(node);
		try {
			return supplier.get();
		} finally {
			leave(node);
		}
	}

	
	protected static String stateToString() {
		if (nodes.get()==null) {
			return "()";
		}
		StringBuilder sb = new StringBuilder("(");
		for (String node:nodes.get()) {
			if (sb.length()>1) {
				sb.append(" > ");
			}
			sb.append(node);
		}
		sb.append(")");
		return sb.toString();
	}
	
	/**
	 * Returns depth of current traversal.
	 */
	public static int depth() {
		return nodes.get()==null? 0:nodes.get().size();
	}
}
