/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.svncr;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.components.provider.IComponentsRepositoryContext;
import com.zfabrik.components.provider.util.AbstractComponentRepository;
import com.zfabrik.components.provider.util.FSCRDBComponent;
import com.zfabrik.components.provider.util.OfflineModeException;
import com.zfabrik.svnaccess.IDirEntryHandler;
import com.zfabrik.svnaccess.IStreamHandler;
import com.zfabrik.svnaccess.ISvnLogEntryHandler;
import com.zfabrik.svnaccess.ISvnRepository;
import com.zfabrik.svnaccess.NodeKind;
import com.zfabrik.svnaccess.SvnDirEntry;
import com.zfabrik.svnaccess.SvnInfo;
import com.zfabrik.svnaccess.SvnLogItem;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.work.WorkUnit;

/**
 * Subversion based components repository based on
 * {@link AbstractComponentRepository}
 * <p>
 * Revision determination is a little tricky and essentially based on
 * max(package rev, component rev) an approx of the pegrevision
 * (http://svnbook.red-bean.com/nightly/en/svn.advanced.pegrevs.html) with a
 * non-trivial scheme for delta updates. The reason for doing this is
 * subversions preservation of revisions of nested objects in case of moves
 * (renames) and branch copies.
 * <p>
 * Repository configuration (i.e. component level) properties:
 * <table>
 * <tr>
 * <td>svncr.url</td>
 * <td>URL of the subversion root folder of the repository. E.g. something like
 * <code>svn://z2-environment.net/z2_base/trunk/l1</code></td>
 * </tr>
 * <tr>
 * <td>svncr.user</td>
 * <td>User name for Subversion authentication (optional)</td>
 * </tr>
 * <tr>
 * <td>svncr.password</td>
 * <td>Password for Subversion authentication (optional)</td>
 * </tr>
 * <tr>
 * <td>svncr.priority</td>
 * <td>Priority of the repository with the repository chain. See
 * {@link IComponentsRepository}.</td>
 * </tr>
 * </table>
 *
 * <p>
 * System properties that have a meaning:
 * <table>
 * <tr>
 * <td><code>com.zfabrik.svncr.mode</code></td>
 * <td>(DEPRECATED) If set to "relaxed" will only issue a warning when repository access
 * fails during sync revision lookup (e.g. on startup) - i.e. if the repo
 * revision cannot be determined via a repository request. This means that a
 * given system may continue to run although a repository may not be accessible,
 * as long no requests have to made because resources are not (yet) available in
 * the repository cache. Defaults to <code>strict</code></td>
 * </tr>
 * </table>
 *
 * @author hb
 *
 */
public class ComponentRepositoryImpl extends AbstractComponentRepository<ComponentRepositoryDB> {
	/**
	 * If set to "relaxed" will only issue a warning when repository access
	 * fails during sync revision lookup (e.g. on startup) - i.e. if the repo
	 * revision cannot be determined via a repository request. This means that a
	 * given system may continue to run although a repository may not be
	 * accessible, as long no requests have to made because resources are not
	 * (yet) available in the repository cache. Defaults to <code>strict</code>
	 *
	 * @Deprecated Please use {@link IComponentsRepository#COMPONENT_REPO_MODE}
	 */
	public final static String MODE = "com.zfabrik.svncr.mode";
	/**
	 * Permissable value of {@link #MODE}
	 * @Deprecated Please use {@link IComponentsRepository#COMPONENT_REPO_MODE}
	 */
	public final static String MODE_RELAXED = "relaxed";
	/**
	 * Permissable value of {@link #MODE}
	 * @Deprecated Please use {@link IComponentsRepository#COMPONENT_REPO_MODE}
	 */
	public final static String MODE_STRICT = "strict";
	/**
	 * URL of the subversion root folder for this repository.
	 */
	public final static String SVNCR_URL = "svncr.url";
	/**
	 * User name for authentication at the repository
	 */
	public final static String SVNCR_USER = "svncr.user";
	/**
	 * Password for authentication at the repository
	 */
	public final static String SVNCR_PWD = "svncr.password";
	/**
	 * priority of the repository with the repository chain. See
	 * {@link IComponentsRepository}.
	 */
	public final static String SVNCR_PRIO = "svncr.priority";
	/**
	 * Folder name in cache. Can and should be omitted in most cases. If
	 * omitted, will be computed.
	 */
	public final static String SVNCR_ROOT = "svncr.rootFolder";

	private final static String PROPS_SUFFIX = ".properties";

	// the uuid of the repo the component is from
	private final static String REPO_UUID = "com.zfabrik.svncr.uuid";

    private static final String SVN_ADAPTER_COMPONENT = "com.zfabrik.boot.svnaccess/svnAccessor";

    private String url, username, password, uuid;
	private boolean relaxed = super.isRelaxedMode() || MODE_RELAXED.equalsIgnoreCase(System.getProperty(MODE));

	/**
	 * Expected repository configuration. These properties are used to
	 * compare for repo purging (see #1879)
	 */
	private Properties expectedConfiguration;

	
	// svn access
	private ISvnRepository test_accessor;

	
	
	public ComponentRepositoryImpl(String name, Properties props) {
		super(name, ComponentRepositoryDB.class);

		// read config
		int prio;
		String prios = props.getProperty(SVNCR_PRIO);
		if (prios == null || (prios = prios.trim()).length() == 0) {
			prio = 500;
		} else {
			try {
				prio = Integer.parseInt(prios);
			} catch (NumberFormatException nfe) {
				throw new IllegalStateException("Invalid priority (" + SVNCR_PRIO + ") specification (" + prios + "): " + this.getName(), nfe);
			}
		}
		this.url = normalize(props.getProperty(SVNCR_URL));
		if (this.url == null) {
			throw new IllegalStateException("Missing property " + SVNCR_URL + ": " + this.getName());
		}

		this.username = normalize(props.getProperty(SVNCR_USER));
		this.password = normalize(props.getProperty(SVNCR_PWD));
		
		// compute expected config		
		this.expectedConfiguration = new Properties();
		this.expectedConfiguration.putAll(props);
		
		// remove some config that we consider insignificant for repo purging
		this.expectedConfiguration.remove(SVNCR_PRIO);

		configure(prio);

		if (!Foundation.isWorker()) {
			logger.info("Using " + this.toString());
			if (this.relaxed) {
				logger.warning("SVN Component Repository running in relaxed mode. Do not use in production!");
			}
		}
	}
	
	@Override
	protected Properties getExpectedConfiguration() {
		return this.expectedConfiguration;
	}

	private boolean relaxing(Exception e) {
		if (e instanceof OfflineModeException) {
			logger.warning("System is running in offline mode. Will continue without attempting remote access: " + this.getName());
			return true;
		}
		if (this.relaxed) {
			logger.warning("Caught error during repository access (" + e.toString() + "). Will continue because of relaxed mode: " + this.getName());
			return true;
		}
		return false;
	}

	private static String normalize(String in) {
		if (in != null) {
			in = in.trim();
			if (in.length() > 0) {
				return in;
			}
		}
		return null;
	}

	@Override
	public ComponentRepositoryDB scan(ComponentRepositoryDB current) {
		ComponentRepositoryDB resDB;
		try {
			checkOfflineMode();
			obtainUUID();
			// get latest rev
			long targetRevision = getRepoSession().getRepoRevision();
			if (current == null) {
				// no db - must scan
				resDB = fullScan(targetRevision);
			} else {
				if (!eq(current.getUuid(), uuid)) {
					// changed uuid - must scan
					logger.warning("Found changed subversion UUID (previously " + current.getUuid() + " now " + uuid
							+ ") for repo. Will run full scan: " + this.getName());
					resDB = fullScan(targetRevision);
				} else {
					// updated rev, can run delta scan
					if (targetRevision > current.getRevision()) {
						resDB = deltaScan(current, targetRevision);
					} else {
						// no update needed
						resDB = current;
					}
				}
			}
		} catch (Exception e) {
			if (!relaxing(e) || current == null) {
				throw new RuntimeException("Scan on SVN repo failed: " + this.getName(), e);
			} else {
				// simply keep using the current DB
				resDB = current;
			}
		}

		assert resDB == null || resDB.getUuid() != null;
		return resDB;
	}

	private void obtainUUID() throws IOException {
		// retrieve current uuid
		this.uuid = getRepoSession().getRepositoryUuid();
		if (this.uuid==null) {
			throw new IllegalStateException("Failed to retrieve UUID of repo: "+this.getName());
		}
	}

	private boolean eq(Object o1, Object o2) {
		if (o1 == null) {
			return o2 == null;
		}
		return o1.equals(o2);
	}

	/*
	 * Delta scanning. Update a component db between revisions by applying
	 * changes from an SVN log
	 */
	private ComponentRepositoryDB deltaScan(final ComponentRepositoryDB current, final long targetRevision) throws Exception {
		logger.fine("Running delta scan of SVN Repository for target revision " + targetRevision + ": " + this.getName());

        final SvnWorkResource repoSession = getRepoSession();

        // copy
		final ComponentRepositoryDB db = new ComponentRepositoryDB(current);
		db.setRevision(targetRevision);
		db.setUuid(this.uuid);

		// fetch the log in ascending revision order
		final Map<String, Update> chgMap = new HashMap<String, Update>();
		repoSession.svnLog("", current.getRevision() + 1, targetRevision, new ISvnLogEntryHandler() {
            @Override
            public void handleLogEntry(SvnLogItem logItem) throws Exception {

                String svnUrl = repoSession.getSvnRootUrl() + logItem.getPath();
                String baseUrlSlash = repoSession.getBaseUrl() + "/";

                // log items might not belong to the component repository!
                // Note: the lowercase condition is necessary due to problems mit lower case hostname conversions.
                // It might lead to more invalidations than absolutely necessary - but that seems acceptable
                if (!svnUrl.toLowerCase().startsWith(baseUrlSlash.toLowerCase())) return;

                // crRelPath is path relative to the CR-URL (baseUrl); it doesn't start with '/' thus split() is safe.
                String crRelPath = svnUrl.substring(baseUrlSlash.length());
                String[] segments = crRelPath.split("/");

                String moduleName = segments.length >= 1 ? segments[0] : null;

                if (moduleName != null) {
                    String compName = segments.length >= 2 ? segments[1] : null;

                    if (compName != null) {
                        // it's <module>/<component>

                        if (!chgMap.containsKey(moduleName) && !compName.startsWith(".")) {
                            // consider component-level changes only if there's no corresponding module-level entry

                            int propsPos = compName.indexOf(PROPS_SUFFIX);
                            if (propsPos != -1) {
                                compName = compName.substring(0, propsPos);
                            }
                            // simply memorize the plain component name "m/c"
                            // later we'll care about single-file components vs folder-based components
                            // i.e: m/c.properties vs m/c/z.properties
                            String fullName = moduleName + "/" + compName;
                            chgMap.put(fullName, Update.cmp(chgMap.get(fullName), logItem.getRevision()));
                        }

                    } else {
                        // it's <module>

                        switch (logItem.getAction()) {
                            case deleted:
                                // delete adds a package entry (and only delete is interesting on the package)
                                // previous ADDs will be overridden.
                                chgMap.put(moduleName, Update.pkg(chgMap.get(moduleName), Update.DEL, logItem.getRevision()));
                                // remove all component level entries for this package.
                                removeComponentEntriesForModule(chgMap, moduleName);
                                break;
                            case added:
                                // change to an add (subversion move will only note a DEL and then an ADD. So we need to introspect
                                // previous DELs will be overridden.
                                chgMap.put(moduleName, Update.pkg(chgMap.get(moduleName), Update.ADD, logItem.getRevision()));
                                // remove all corresponding component level entries for this module.
                                removeComponentEntriesForModule(chgMap, moduleName);
                                break;
                            case replaced:
                                throw new IOException("Unexpected action 'replaced' in " + repoSession + " for path " + logItem.getPath() + "@" + logItem.getRevision());
                            default: // ignore
                        }
                    }
                }
            }
        }
        );

		logger.fine("Found " + chgMap.size() + " changes (components or module)");
		// now apply
		for (Map.Entry<String, Update> mu : chgMap.entrySet()) {
			Update u = mu.getValue();
			// mcId is either <module> or <module>/<compName>
			final String mcId = mu.getKey();
			if (u.isPackage) {
				// <module>
				if (u.type == Update.DEL) {
					// remove all components of that module
					db.removeModule(mcId);
				} else if (u.type == Update.ADD) {
					// a package add can be the result of a rename/move, in
					// which case the single contained resources
					// will not be listed in the log again.
					// therefore we need to fetch all components of the package.
					SvnInfo moduleInfo = repoSession.svnInfo(mcId, targetRevision);
					readModuleComponents(targetRevision, db, moduleInfo);
				} else {
					throw new IllegalStateException("No idea how to handle op " + u.type + " on module " + mcId);
				}
			} else {
				// mcId is <module>/<component>
				// try to read the component
				FSCRDBComponent c = new FSCRDBComponent(mcId);
				c.getProperties().setProperty(REPO_UUID, this.uuid);
				// we pull the revision from the update info
				c.setRevision(u.revision);

                try {
				    repoSession.svnInfo(mcId + PROPS_SUFFIX, targetRevision);
					c.setHasFile(true);
				} catch (IOException e) {
                    // ignore exception: it's not a file, maybe it's a folder
                }

                try {
                    repoSession.svnInfo(mcId, targetRevision);
					c.setHasFolder(true);
				} catch (IOException e) {
                    // ignore exception: it's not a folder, maybe it was a file
                }

				if ((!c.isHasFile()) && (!c.isHasFolder())) {
					// not existing. Try remove
					db.removeComponent(mcId);
				} else {
					// read new props and mark as unresolved
					if (readComponentProperties(c, targetRevision)) {
						db.putComponent(mcId, c);
					}
				}
			}
		}
		logger.info("SVN Component Repository now at revision " + db.getRevision() + ": " + this.getName());
		return db;
	}

    private void removeComponentEntriesForModule(Map<String, Update> m, String pa) {
		String prfx = pa + "/";
		Iterator<Map.Entry<String, Update>> i = m.entrySet().iterator();
		while (i.hasNext()) {
			if (i.next().getKey().startsWith(prfx)) {
				i.remove();
			}
		}
	}

	/*
	 * Full scan. Introspect the repository at target revision and build a new
	 * DB
	 */
	private ComponentRepositoryDB fullScan(final long targetRevision) throws Exception {
		logger.fine("Running full scan of SVN Repository for target revision " + targetRevision + ": " + this.getName());
		// full scan
		final ComponentRepositoryDB db = new ComponentRepositoryDB(targetRevision, this.uuid);
		final ISvnRepository repo = getRepoSession().getSVNRepository();

		// read all modules folders.
		int num = getRepoSession().svnList("", targetRevision, new IDirEntryHandler() {
            public void handleSvnDirEntry(SvnDirEntry entry) throws Exception {
                // udoo [13.09.2012] : only inspect directory nodes and ignore files - see http://redmine.z2-environment.net/issues/898
                if (entry.getNodeKind() == NodeKind.dir) {
                    readModuleComponents(targetRevision, db, entry);

                } else if (entry.getNodeKind() != NodeKind.file) {
                    // SVN LOG can return node-kind=="unknown" but we never discovered that for SVN LIST.
                    throw new IllegalStateException("svn list returns " + entry + " with unsupported node-kind for " + repo.getBaseUrl());
                }
            }
        });

		logger.fine("Found " + num + " modules.");
		logger.info("SVN Component Repository now at revision " + db.getRevision() + ": " + this.getName());
		return db;
	}

	/*
	 * Read properties of a single component
	 */
	private boolean readComponentProperties(FSCRDBComponent c, long rev) {
		// read props
		try {
			logger.finer("Reading component properties " + c.getName());
			// The RULE: Folder wins over props
			String zPropsPath = c.getName() + (c.isHasFolder() ? "/z" + PROPS_SUFFIX : PROPS_SUFFIX);
			final Properties p = new Properties();
			getRepoSession().svnCat(zPropsPath, rev, new IStreamHandler() {
                public void handleStream(InputStream ins) throws Exception {
                    p.load(ins); // ins is closed by repo.getContent()
                }
            });
			p.setProperty(COMPONENT_REPO_IMPLEMENTATION, getName());
			c.setProperties(p);
			return true;

		} catch (IOException svne) {
			logger.fine("Component " + c.getName() + " is broken: No properties. Will be ignored");
			return false;
		}
	}

	/*
	 * read component data for a whole module (finds what qualifies as component
	 * and determines its revision and its properties).
	 */
	private void readModuleComponents(final long targetRevision, final ComponentRepositoryDB db, final SvnDirEntry moduleEntry) throws IOException {
        final SvnWorkResource repoSession = getRepoSession();
        final Map<String, FSCRDBComponent> cmps = new HashMap<String, FSCRDBComponent>();
		final String modulePrefix = moduleEntry.getCrRelPath() + "/";

		repoSession.svnList(moduleEntry.getCrRelPath(), targetRevision, new IDirEntryHandler() {

            public void handleSvnDirEntry(SvnDirEntry entry) throws Exception {
                // ignore components starting with "."
                String path = entry.getPath();
                if (!path.startsWith(".")) {

                    switch (entry.getNodeKind()) {
                        case dir: {
                            // folder-based components with "z.properties" inside - here we simply collect all folders including non-component folders
                            // Note: if this is not a component folder (i.e. no z.properties inside) it will be skipped later when the properties are read.
                            String compId = modulePrefix + path;
                            FSCRDBComponent t = cmps.get(compId);
                            if (t == null) {
                                t = new FSCRDBComponent(compId);
                                // set the uuid so in case it changed, the component change will be noted
                                t.getProperties().setProperty(REPO_UUID, repoSession.getRepositoryUuid());
                                cmps.put(compId, t);
                            }
                            t.setHasFolder(true);
                            t.setRevision(Math.max(entry.getRevision(), moduleEntry.getRevision()));
                            break;
                        }

                        case file: {
                            // single-file component like "datasource.properties"
                            if (path.endsWith(PROPS_SUFFIX)) {

                                // compId is module "/" component (w/o .properties)
                                String compId = modulePrefix + path.substring(0, path.length() - PROPS_SUFFIX.length());

                                FSCRDBComponent t = cmps.get(compId);
                                if (t == null) {
                                    t = new FSCRDBComponent(compId);
                                    // set the uuid so in case it changed, the component change will be noted
                                    t.getProperties().setProperty(REPO_UUID, repoSession.getRepositoryUuid());
                                    cmps.put(compId, t);
                                }
                                t.setHasFile(true);
                                if (!t.isHasFolder()) {
                                    // folder rules!
                                    // we max package and component
                                    // NOTE: subversion move preserves inner versions (i.e. this is something like a peg revision)
                                    t.setRevision(Math.max(entry.getRevision(), moduleEntry.getRevision()));
                                }
                            }
                            break;
                        }

                        default:
                            // SVN LOG can return node-kind=="unknown" but we never discovered that for SVN LIST.
                            throw new IllegalStateException("svn list returns " + entry + " with unsupported node-kind for " + repoSession.getBaseUrl());

                    }
                }
            }
        });

		// read the props (folder w/o "z.properties" will be ignored)
		for (Map.Entry<String, FSCRDBComponent> e : cmps.entrySet()) {
			if (readComponentProperties(e.getValue(), targetRevision)) {
				db.putComponent(e.getKey(), e.getValue());
			}
		}
	}

	/*
	 * Download a component's resources into the given folder
	 */
	@Override
	public void download(FSCRDBComponent c, File destFolder) {
		// we only care about components with folders
		if (! c.isHasFolder()) return;
		try {
			logger.fine("Downloading component " + c.getName() + ": " + this.getName());
			// don't continue in offline mode!
			checkOfflineMode();
			long start = System.currentTimeMillis();

			getRepoSession().svnExport(c.getName(), getRepoSession().getRepoRevision(), destFolder);
			long duration = System.currentTimeMillis() - start;
			logger.fine("Download of component " + c.getName() + " completed after " + duration + "ms");
		} catch (OfflineModeException ex) {
			throw new RuntimeException("Resources for component "+c.getName()+" are not available locally and cannot be retrieved as we are running in offline mode");
		} catch (IOException ex) {
			throw new RuntimeException("Repository access problem", ex);
		}
	}

	@Override
	protected String getURL() {
		return this.url;
	}

	@Override
	protected long getRevision() {
		try {
			ComponentRepositoryDB d = getDB();
			return d.getRevision();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	//
	// util
	//
	private static class Update {
		static final char NIL = 0;
		static final char CMP = 'M';
		static final char DEL = 'D';
		static final char ADD = 'A';

		long revision;
		boolean isPackage;
		char type = NIL;

		private Update() {
		}

		static Update cmp(Update u, long revision) {
			if (u == null)
				u = new Update();
			u.type = CMP;
			u.isPackage = false;
			u.revision = revision;
			return u;
		}

		static Update pkg(Update u, char type, long revision) {
			if (u == null)
				u = new Update();
			u.isPackage = true;
			u.type = type;
			u.revision = revision;
			return u;
		}

		@Override
		public String toString() {
			return type + (isPackage ? "[pkg]" : "") + "@" + revision;
		}
	}

	//
	// svn specific handling
	//

	// accessing the repo.
	public synchronized SvnWorkResource getRepoSession() {
		String key = SvnWorkResource.class.getName() + "/" + this.getName();
		SvnWorkResource svnwr = (SvnWorkResource) WorkUnit.getCurrent().getResource(key);
		if (svnwr == null) {
			// in test cases we do not rely on a resolvable env.
            ISvnRepository svnClient = test_accessor != null? test_accessor : IComponentsLookup.INSTANCE.lookup(SVN_ADAPTER_COMPONENT, ISvnRepository.class);
            svnClient.setBaseUrl(this.url);
            svnClient.setUsername(this.username);
            svnClient.setPassword(this.password);
			svnwr = new SvnWorkResource(svnClient);
			try {
				WorkUnit.getCurrent().bindResource(key, svnwr);
			} catch (Exception e) {
				throw new IllegalStateException("Failed to bind SVNCR work resource", e);
			}
		}
		return svnwr;
	}

	@Override
	public String toString() {
		return "SVN-CR: " + super.toString() + ", url:" + this.url + ", username:" + this.username;
	}

	private final static Logger logger = Logger.getLogger(ComponentRepositoryImpl.class.getName());

	public static ComponentRepositoryImpl test_create(String name, File root, Properties props, ISvnRepository svnRepository) {
//		System.setProperty(Foundation.HOME_LAYOUT_REPOS, root.getAbsolutePath());
        ComponentRepositoryImpl c = new ComponentRepositoryImpl(name, props);
		c.test_setRepository(svnRepository);
		c.test_setInited(true);
		c.test_setContext(new IComponentsRepositoryContext() {
			public IComponentsRepository next() {
				return null;
			}
		});
		return c;
	}

	public void test_setRepository(ISvnRepository accessor) {
		this.test_accessor = accessor;
	}

	public ISvnRepository test_getRepo() throws IOException {
		return getRepoSession().getSVNRepository();
	}

	public ComponentRepositoryDB test_fullScan(long rev) throws Exception {
		obtainUUID();
		return fullScan(rev);
	}

	public ComponentRepositoryDB test_deltaScan(ComponentRepositoryDB db, long rev) throws Exception {
		obtainUUID();
		return deltaScan(db, rev);
	}

}
