/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.management.jmx;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.management.ManagementFactory;
import java.nio.channels.FileLock;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.management.openmbean.CompositeData;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.IDependencyComponent;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.util.threading.TimerUtil;

public class JmxLoggerResource extends Resource implements IDependencyComponent {
	private final static String BASE_NAME = "jmxlogging.file.baseName";
	private final static String INTERVAL = "jmxlogging.interval";
	// attribute keys a1,a2,...
	private final static String ATTRIBUTES = "jmxlogging.attributes";
	private final static String ATTRIBUTE_PREFIX = "jmxlogging.attribute.";
	private final static String MBEAN_SUFFIX = ".mbean";
	private final static String ATTRIBUTE_SUFFIX = ".attribute";

	private String name;
	private String baseName;
	private File file;
	private int dayOfYear = -1;

	private String[] fields;
	private Map<String, ObjectName> mbeans;
	private Map<String, String> attributes;
	private int intervalS;
	private int interval;
	private Timer timer;

	// the actual reporting
	private class ReporterTask extends TimerTask {
		private MBeanServer mbs;

		public ReporterTask() {
			this.mbs = ManagementFactory.getPlatformMBeanServer();
		}

		public void run() {
			try {
				_checkFile();
				RandomAccessFile raf = new RandomAccessFile(file, "rw");
				FileLock lock = raf.getChannel().lock();
				StringBuilder b = new StringBuilder(fields.length * 10);
				try {
					raf.seek(raf.length());
					if (raf.getFilePointer() == 0) {
						// start a new one with a headline
						b.append("t");
						for (int i = 0; i < fields.length; i++) {
							b.append(",");
							b.append(fields[i]);
						}
						b.append("\n");
						raf.writeChars(b.toString());
						b.setLength(0);
					}
					String fn, at;
					ObjectName on;
					int a, e;
					b.append(System.currentTimeMillis());
					for (int i = 0; i < fields.length; i++) {
						b.append(",");
						fn = fields[i];
						at = attributes.get(fn);
						on = mbeans.get(fn);
						a = 0;
						try {
							e = at.indexOf('/', a);
							if (e < 0) {
								Object o = mbs.getAttribute(on, at);
								if (o == null) {
									b.append("ERR404");
								}
								b.append(o.toString());
							} else {
								CompositeData cd = null;
								String an;
								while (e >= 0) {
									an = at.substring(a, e);
									cd = (CompositeData) mbs.getAttribute(on, an);
									a = e + 1;
									e = fn.indexOf('/', a);
								}
								Object o = cd.get(at.substring(a));
								if (o == null) {
									b.append("ERR404");
								}
								b.append(o.toString());
							}
						} catch (AttributeNotFoundException ex) {
							b.append("ERR404");
						} catch (InstanceNotFoundException ex) {
							b.append("ERR404");
						} catch (ClassCastException ex) {
							b.append("ERR405");
						} catch (MBeanException ex) {
							b.append("ERR500");
						} catch (ReflectionException ex) {
							b.append("ERR500");
						}
					}
					b.append("\n");
					raf.writeChars(b.toString());
				} finally {
					lock.release();
					raf.close();
				}
			} catch (IOException ioe) {
				logger.log(Level.SEVERE, "Exception caught in JMX reporting task", ioe);
			}
		}
	}

	private void _checkFile() {
		Calendar now = Calendar.getInstance();
		if ((this.file == null) || (this.dayOfYear != now.get(Calendar.DAY_OF_YEAR))) {
			String fn = String.format("%s_%04d%02d%02d_%s.csv", this.baseName, now.get(Calendar.YEAR), now.get(Calendar.MONTH) + 1, now
					.get(Calendar.DAY_OF_MONTH), (Foundation.isWorker() ? Foundation.getProperties().getProperty(Foundation.PROCESS_WORKER).replace(
					"/", "_") : "home"));
			File jmxfo = new File("../jmx");
			if (!jmxfo.exists()) {
				jmxfo.mkdirs();
			}
			this.file = new File(jmxfo, fn);
			this.dayOfYear = now.get(Calendar.DAY_OF_YEAR);
		}
	}

	//

	public JmxLoggerResource(String name) {
		this.name = name;
	}

	public void invalidate() throws ResourceBusyException {
		stop();
	}

	public <T> T as(Class<T> clz) {
		if (IDependencyComponent.class.equals(clz)) {
			return clz.cast(this);
		}
		return null;
	}

	private synchronized void stop() {
		try {
			if (this.timer != null) {
				logger.info("Stopping JMX log update: " + this.name);
				this.timer.cancel();
			}
		} finally {
			this.timer = null;
			this.mbeans = null;
			this.attributes = null;
			this.fields = null;
		}
	}

	public void prepare() {
		start();
	}

	private synchronized void start() {
		if (this.timer == null) {
			IComponentDescriptor desc = IComponentsManager.INSTANCE.getComponent(this.name);
			Properties p = desc.getProperties();

			// frequency (overridable)
			String fqs = desc.getProperty(INTERVAL);
			try {
				this.intervalS = Integer.parseInt(fqs);
				this.interval = this.intervalS * 1000;
			} catch (NumberFormatException nfe) {
				throw new IllegalStateException("Failed to parse reporting frequency (" + INTERVAL + "): " + this.name, nfe);
			}

			// file
			String fn = p.getProperty(BASE_NAME);
			if (fn == null) {
				fn = "jmx";
			}
			this.baseName = fn;
			// get all fields
			String fs = p.getProperty(ATTRIBUTES);
			if (fs == null) {
				throw new IllegalStateException("Component descriptor specifies no monitorable attributes (" + ATTRIBUTES + "): " + this.name);
			}
			StringTokenizer tk = new StringTokenizer(fs, ",");
			this.fields = new String[tk.countTokens()];
			this.attributes = new HashMap<String, String>(this.fields.length);
			this.mbeans = new HashMap<String, ObjectName>(this.fields.length);
			String fi, on, at;
			int i = 0;
			while (tk.hasMoreTokens()) {
				fi = tk.nextToken().trim();
				this.fields[i++] = fi;
				if (this.attributes.containsKey(fi)) {
					throw new IllegalStateException("Duplicate attribute specified (" + ATTRIBUTES + "): " + this.name);
				}
				on = p.getProperty(ATTRIBUTE_PREFIX + fi + MBEAN_SUFFIX);
				if (on == null) {
					throw new IllegalStateException("No object name specified for attribute (" + (ATTRIBUTE_PREFIX + fi + MBEAN_SUFFIX) + "): "
							+ this.name);
				}
				try {
					this.mbeans.put(fi, ObjectName.getInstance(on.trim()));
				} catch (MalformedObjectNameException e) {
					throw new IllegalStateException("Failed to parse object name specification (" + (ATTRIBUTE_PREFIX + fi + MBEAN_SUFFIX) + "): "
							+ this.name, e);
				}
				at = p.getProperty(ATTRIBUTE_PREFIX + fi + ATTRIBUTE_SUFFIX);
				if (at == null) {
					throw new IllegalStateException("No MBean attribute specified for attribute (" + (ATTRIBUTE_PREFIX + fi + ATTRIBUTE_SUFFIX)
							+ "): " + this.name);
				}
				this.attributes.put(fi, at.trim());
			}
			logger.info("Scheduling JMX log update (baseName: " + this.baseName + ")" + " for every " + this.intervalS + "s: " + this.name);
			this.timer = TimerUtil.createTimer("JMX log writer (" + this.baseName + ")", true);
			this.timer.scheduleAtFixedRate(new ReporterTask(), this.interval, this.interval);
		}
	}

	private final static Logger logger = Logger.getLogger(JmxLoggerResource.class.getName());
}
