/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.hubcr;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Base64;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import com.zfabrik.components.provider.util.FSCRDBComponent;
import com.zfabrik.hubcr.RemoteComponentRepositoryDB;
import com.zfabrik.util.html.Escaper;

public class RemoteClient {
	private String baseUrl;
	private String user, password;
	private long timeout;

	public RemoteClient(String url, String user, String password, long timeout) {
		if (!url.toLowerCase().startsWith("http")) {
			throw new IllegalArgumentException("Currently only supporting HTTP urls");
		}
		if (!url.endsWith("/")) {
			url += "/";
		}
		this.baseUrl = url+"repo";
		this.user = user;
		this.password = password;
		this.timeout = timeout;
	}

	/**
	 * Download a single component's resources. 
	 * 
	 * <p>
	 * The used URL is 
	 * <p>
	 * &lt;base&gt;/z_remotecr/download?component=&lt;component&gt;&amp;&lt;rev&gt;
	 * <p>
	 * 
	 * 
	 * @param component
	 * @param folder
	 */
	public void download(FSCRDBComponent component, File folder) throws IOException {
		URL u = new URL(
			new StringBuilder(this.baseUrl)
			.append("/download?componentName=")
			.append(Escaper.urlEncode(component.getName()))
			.append("&rev=")
			.append(Long.toString(component.getRevision()))
			.toString()
		);
		HttpURLConnection c = (HttpURLConnection) u.openConnection();
		c.setConnectTimeout((int) timeout);
		authorize(c);
		c.setDoOutput(false);
		c.setDoInput(true);
		c.setRequestMethod("GET");
		byte[] buffer = new byte[16384];
		int l;
		int rc = c.getResponseCode();
		if (rc!=200) {
			String msg = c.getResponseMessage();
			throw new RuntimeException("Remote repository returned "+rc+(msg!=null? " ("+msg+")":""));
		}
		// read
		ZipInputStream zIn = new ZipInputStream(c.getInputStream());
		try {
			ZipEntry ze;
			while ((ze = zIn.getNextEntry())!=null) {
				String n = ze.getName();
				if (n.endsWith("/")) {
					// folder
					n = n.substring(0,n.length()-1);
					new File(folder,n).mkdirs();
				} else {
					// regular file
					File f = new File(folder,n);
					f.getParentFile().mkdirs();
					OutputStream out = new FileOutputStream(f);
					try { 
						while ((l=zIn.read(buffer))>=0) { out.write(buffer,0,l); } 
					} finally { 
						out.close(); 
					}
				}
			}
		} finally {
			zIn.close();
		}
	}

	/**
	 * Fetch a DB revision from remote. Returns <code>null</code>, if there is no more up 
	 * to date revision available than the one passed in.
	 * <p>
	 * The used URL is 
	 * <p>
	 * &lt;base&gt;/z_remotecr/getDB?rev=&lt;rev&gt;
	 * <p>
	 * The remote side is expected to either deliver a serialized {@link RemoteComponentRepositoryDB} or a status code 304 (not modified).
	 * 
	 * @param rev Revision requested
	 * @return DB instance
	 */
	public RemoteComponentRepositoryDB fetchDBUpdate(Long rev) throws Exception {
		StringBuilder sb = new StringBuilder(this.baseUrl).append("/getDB");
		if (rev!=null) {
			sb.append("?rev=")
			.append(rev.toString());
		}
		URL u = new URL(sb.toString());
		HttpURLConnection c = (HttpURLConnection) u.openConnection();
		c.setConnectTimeout((int) timeout);
		authorize(c);
		c.setDoOutput(false);
		c.setDoInput(true);
		c.setRequestMethod("GET");
		c.connect();
		int rc = c.getResponseCode();
		if (rc==304) {
			// not modified indicates no later DB
			return null;
		}
		if (rc!=200) {
			String msg = c.getResponseMessage();
			throw new RuntimeException("Remote repository returned "+rc+(msg!=null? " ("+msg+")":""));
		}
		// fetch and de-serialize
		ObjectInputStream oIn = new ObjectInputStream(c.getInputStream()) {
			protected Class<?> resolveClass(ObjectStreamClass desc) throws IOException, ClassNotFoundException {
				return Class.forName(desc.getName(), false, RemoteClient.class.getClassLoader());
			}
		};
		try {
			return (RemoteComponentRepositoryDB) oIn.readObject();
		} finally {
			oIn.close();
		}
	}

	private void authorize(HttpURLConnection c) {
		if (this.user!=null) {
			String up = this.user+":"+this.password;
			String ec = new String(Base64.getEncoder().encode(up.getBytes()));
			c.setRequestProperty("Authorization", "Basic "+ec);
		}
	}

}
