/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.svnaccess;

import static com.zfabrik.impl.svnaccess.PathHelper.concat;
import static com.zfabrik.impl.svnaccess.PathHelper.deSlashEnd;
import static com.zfabrik.impl.svnaccess.PathHelper.deSlashStart;
import static com.zfabrik.impl.svnaccess.PathHelper.relativize;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.subversion.javahl.ClientException;
import org.apache.subversion.javahl.ISVNClient;
import org.apache.subversion.javahl.SVNClient;
import org.apache.subversion.javahl.callback.InfoCallback;
import org.apache.subversion.javahl.callback.ListCallback;
import org.apache.subversion.javahl.callback.LogMessageCallback;
import org.apache.subversion.javahl.types.ChangePath;
import org.apache.subversion.javahl.types.Depth;
import org.apache.subversion.javahl.types.DirEntry;
import org.apache.subversion.javahl.types.Info;
import org.apache.subversion.javahl.types.Lock;
import org.apache.subversion.javahl.types.Revision;
import org.apache.subversion.javahl.types.RevisionRange;

import com.zfabrik.svnaccess.IDirEntryHandler;
import com.zfabrik.svnaccess.IStreamHandler;
import com.zfabrik.svnaccess.ISvnLogEntryHandler;
import com.zfabrik.svnaccess.ISvnRepository;
import com.zfabrik.svnaccess.NodeKind;
import com.zfabrik.svnaccess.SvnDirEntry;
import com.zfabrik.svnaccess.SvnInfo;
import com.zfabrik.svnaccess.SvnLogItem;
import com.zfabrik.svnaccess.SvnLogItem.Action;


/**
 * Implementation of {@link ISvnRepository} over JavaHL. This class depends on the JavaHL API and may not be loaded
 * unless that is present.
 *
 * A word on paths:
 * <ul>
 * <li>In CR paths don't have leading slashes</li>
 * <li>Repo URLs never have trailing slashes</li>
 * </ul>
 */
public class JavaHLSvnRepository implements ISvnRepository {
	private final static Logger LOG = Logger.getLogger(JavaHLSvnRepository.class.getName());
    private String baseUrl;
    private Info baseInfo;
    // Java HL client instance
    private ISVNClient svnClient;


    public JavaHLSvnRepository() {
		this.svnClient = new SVNClient();
	}

	@Override
	public String getBaseUrl() {
		return this.baseUrl;
	}

	@Override
	public void setBaseUrl(String baseUrl) {
		this.baseUrl = deSlashEnd(baseUrl);
    	LOG.fine("Creating "+this);
	}

	@Override
	public void setUsername(String username) {
		this.svnClient.username(username);
	}

	@Override
	public void setPassword(String password) {
		this.svnClient.password(password);
	}

	@Override
	public String getRepositoryUuid() throws IOException {
		return baseInfo().getReposUUID();
	}

	@Override
	public long getCurrentCRRevision() throws IOException {
		return baseInfo().getRev();
	}

	@Override
	public String getSvnRootUrl() throws IOException {
		return baseInfo.getReposRootUrl();
	}

	@Override
	public String getCRPath() throws IOException {
		return baseInfo.getPath();
	}

	/**
	 * Info on path in CR
	 */
	@Override
	public SvnInfo info(final String path, final long pegRevision) throws IOException {
		if (LOG.isLoggable(Level.FINE)) {
			LOG.fine(String.format("Info %s %d",path,pegRevision));
		}
		final Info[] i = new Info[1];
		try {
			this.svnClient.info2(
				concat(this.baseUrl,path),
				revision(pegRevision),
				revision(pegRevision),
				Depth.empty,
				null,
				new InfoCallback() {
					@Override
					public void singleInfo(Info info) {
						i[0]=info;
					}
			});
		} catch (ClientException e) {
			throw new IOException(e);
		}
		if (i[0]==null) {
			return null;
		}
		return svnInfo(this.baseUrl,i[0]);
	}

	@Override
	public SvnInfo info() throws IOException {
		return svnInfo(this.baseUrl,baseInfo());
	}

	/**
	 * Log on path in CR
	 */
	@Override
	public int log(String path, long revision, long revisionFrom, long revisionTo, final ISvnLogEntryHandler logEntryHandler) throws IOException {
		LOG.fine(String.format("log %s %d %d",path,revisionFrom,revisionTo));
		try {
			final int[] count = new int[1];
			this.svnClient.logMessages(
				concat(this.baseUrl,path),
				revision(revision),
				Arrays.asList(new RevisionRange(revision(revisionFrom), revision(revisionTo))),
				false,
				true,
				true,
				Collections.<String>emptySet(),
				0,
				new LogMessageCallback() {
					@Override
					public void singleMessage(Set<ChangePath> changedPaths, long revision,	Map<String, byte[]> revprops, boolean hasChildren) {
						if (changedPaths!=null) {
							for (ChangePath cp : changedPaths) {
								Action a = null;
								switch (cp.getAction()) {
									case add: a = Action.added; break;
									case delete: a = Action.deleted; break;
									case modify: a = Action.modified; break;
									case replace: a = Action.replaced; break;
								}
								SvnLogItem li = new SvnLogItem(cp.getPath(), revision, a);
								try {
									logEntryHandler.handleLogEntry(li);
								} catch (Exception e) {
									throw new RuntimeException(e);
								}
								count[0]++;
							}
						}
					}
				}
			);
			return count[0];
		} catch (ClientException e) {
			throw new IOException(e);
		}
	}

	/**
	 * List relative to CR
	 */
	@Override
	public int list(String path, long pegRevision, long revision, final IDirEntryHandler dirEntryHandler) throws IOException {
		LOG.fine(String.format("List %s %d",path,pegRevision));

		try {
			final int[] count = new int[1];
			this.svnClient.list(
				concat(this.baseUrl,path),
				revision(revision),
				revision(pegRevision),
				Depth.immediates,
				DirEntry.Fields.lastChangeRevision,
				false,
				new ListCallback() {
					@Override
					public void doEntry(DirEntry dirent, Lock lock) {
						// ignore the folder itself
						if (dirent!=null && dirent.getPath().length()>0) {
							SvnDirEntry di = new SvnDirEntry(
								// local name
								dirent.getPath(),
								// cr path
								deSlashStart(dirent.getPath()),
								dirent.getLastChangedRevision().getNumber(),
								nodeKind(dirent.getNodeKind())
							);
							try {
								dirEntryHandler.handleSvnDirEntry(di);
							} catch (Exception e) {
								throw new RuntimeException(e);
							}
							count[0]++;
						}
					}
				}
			);
			return count[0];
		} catch (ClientException e) {
			throw new IOException(e);
		}
	}

	/**
	 * Retrieve relative to CR
	 */
	@Override
	public void getContent(String path, long pegRevision, long revision, IStreamHandler streamHandler) throws IOException {
		if (LOG.isLoggable(Level.FINE)) {
			LOG.fine(String.format("getContent %s %d",path,pegRevision));
		}

		try {
			byte[] content = this.svnClient.fileContent(concat(this.baseUrl,path), revision(revision), revision(pegRevision));
			if (content!=null) {
				streamHandler.handleStream(new ByteArrayInputStream(content));
			}
		} catch (Exception e) {
			throw new IOException(e);
		}
	}


	/**
	 * Export relative to CR
	 */
	@Override
	public void export(String path, long revision, File targetDir) throws IOException {
		LOG.fine(String.format("Export %s %d",path,revision));

		try {
			this.svnClient.doExport(
				concat(this.baseUrl,path),
				targetDir.getAbsolutePath(),
				revision(revision),
				Revision.HEAD,
				true,
				true,
				Depth.infinity,
				"\n"
			);
		} catch (Exception e) {
			throw new IOException(e);
		}
	}

	/**
	 * Close the client instance
	 */
	@Override
	public void close() throws IOException {
		if (this.svnClient!=null) {
			try {
				LOG.fine("Closing "+this);
				this.svnClient.dispose();
			} finally {
				this.svnClient=null;
			}
		}
	}


	/**
	 * Turn JavaHL info into svncr info
	 */
	private static SvnInfo svnInfo(String baseUrl, Info info) {
		String url 		= info.getUrl();
		String crpath 	= relativize(url,baseUrl);
		String path     = relativize(url, info.getReposRootUrl());

		return new SvnInfo(
			info.getReposRootUrl(),
			path,
			crpath,
			info.getLastChangedRev(),
			nodeKind(info.getKind()),
			info.getReposUUID()
		);
	}


	/**
	 * Turn JavaHL nodekind into svncr nodekind
	 */
	private static NodeKind nodeKind(org.apache.subversion.javahl.types.NodeKind kind) {
		switch (kind) {
		case dir: return NodeKind.dir;
		case file: return NodeKind.file;
		case none : return NodeKind.none;
		case unknown: return NodeKind.unknown;
		default: throw new IllegalArgumentException();
		}
	}

	/**
	 * baseinfo is retrieved once per client session.
	 */
	private Info baseInfo() throws IOException {
		if (this.baseInfo==null) {
			try {
				this.svnClient.info2(this.baseUrl, Revision.HEAD, Revision.HEAD, Depth.empty, null, new InfoCallback() {
					@Override
					public void singleInfo(Info info) {
						baseInfo = info;
					}
				});
			} catch (ClientException e) {
				throw new IOException(e);
			}
		}
		return baseInfo;
	}

	/**
	 * Compute JavaHL revision for svncr revision (HEAD==-1)
	 */
	private static Revision revision(long r) {
		if (r==HEAD) {
			return Revision.HEAD;
		}
		return Revision.getInstance(r);
	}


	//
	public static void main(String[] args) throws Exception {

		final String baseUrl = "http://z2-environment.net/svn/z2-core.testdata/trunk/data/testrepo";

		SVNClient c = new SVNClient();
		c.username("z2-environment");
		c.password("z2-environment");

		c.info2(baseUrl, Revision.HEAD, Revision.HEAD, Depth.empty, null, new InfoCallback() {

			@Override
			public void singleInfo(Info info) {

				System.err.format(
					  "info.getLastChangedAuthor(): %s\n"
					+ "info.getLastChangedDate(): %s\n"
					+ "info.getLastChangedRev(): %d\n"
					+ "info.getPath(): %s\n"
					+ "info.getReposRootUrl(): %s\n"
					+ "info.getReposSize(): %d\n"
					+ "info.getReposUUID(): %s\n"
					+ "info.getRev(): %d\n"
					+ "info.getUrl(): %s\n"
					,
					info.getLastChangedAuthor(),
					info.getLastChangedDate(),
					info.getLastChangedRev(),
					info.getPath(),
					info.getReposRootUrl(),
					info.getReposSize(),
					info.getReposUUID(),
					info.getRev(),
					info.getUrl()
				);


				System.err.println(svnInfo(baseUrl, info));
			}
		});


		c.list(baseUrl, Revision.HEAD, Revision.HEAD, Depth.immediates, 0 , false, new ListCallback() {

			@Override
			public void doEntry(DirEntry dirent, Lock lock) {
				System.err.format("%s, %s, %s\n", dirent.getAbsPath(), dirent.getPath(), dirent.getNodeKind());
			}
		});

//		c.logMessages(
//			baseUrl,
//			Revision.HEAD,
//			Arrays.asList(new RevisionRange(Revision.getInstance(1), Revision.HEAD)),
//			false,
//			true,
//			true,
//			new HashSet<String>(Arrays.asList("svn:log")),
//			0,
//			new LogMessageCallback() {
//				@Override
//				public void singleMessage(Set<ChangePath> changedPaths, long revision,	Map<String, byte[]> revprops, boolean hasChildren) {
//					System.err.format("\nRev %d:\n---\n%s\n----\n",revision,new String(revprops.get("svn:log")));
//					for (ChangePath cp : changedPaths) {
//						System.err.format("%s %s\n",cp.getAction(),cp.getPath());
//					}
//				}
//			}
//		);


		c.dispose();
	}

	@Override
	public String toString() {
		return "JavaHLSvnClient [baseUrl=" + baseUrl + "]";
	}

}
