/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components;

import java.util.Properties;

import com.zfabrik.components.provider.IComponentDescriptorProcessor;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.resources.provider.Resource;

/**
 * The Component Descriptor provides access to meta data as provided by
 * component repositories ({@link IComponentsRepository} without requiring download of further component
 * resources or other processing.
 * <p>
 * A Component Descriptor can be retrieved by calling {@link IComponentsManager#getComponent(String)} or
 * by a component lookup passing this interface as expected result type:
 * <p>
 * <code>
 * IComponentsLookup.INSTANCE.lookup(&lt;component name&gt;,IComponentDescriptor.class);
 * </code>
 * Component metadata is provided as properties in the format of a {@link Properties} properties file. At least such property file should
 * declare a component type:
 * <pre>
 * com.zfabrik.component.type=&lt;component type&gt;
 * </pre>
 * For example
 * <pre>
 * com.zfabrik.component.type=com.zfabrik.java
 * </pre>
 * would define a Java component.
 * <p>
 * Component property values may be provided dynamically through implementations of {@link IComponentDescriptorProcessor}.
 * Given a component property <code>someNameSpace.somePropertyName</code>, it may be further qualified with a processing
 * style, such as JEXL3, by appending the style to the property name separated by a colon (that needs to be escaped in property
 * file format). In the example this would look as follows:
 * <pre>
 * <code>
 * someNameSpace.somePropertyName\:JEXL3=....
 * </code>
 * </pre>
 * 
 * See the JEXL3 component descriptor processor documentation for more details.
 *
 * @author hb
 * @see IComponentsRepository
 * @see IComponentsLookup
 *
 */

public interface IComponentDescriptor {

	/**
	 * The name of the component. This property can be used conveniently to convey a component name,
	 * e.g. on a servlet request or elsewhere.
	 */
	final static String COMPONENT_NAME = "com.zfabrik.component.name";
	/**
	 * The type of a component. For example <code>com.zfabrik.java</code> for a Java component.
	 */
	final static String COMPONENT_TYPE = "com.zfabrik.component.type";

	/**
	 * The default expression evaluation style for the component descriptor and default value
	 * of {@link #COMPONENT_DESCRIPTOR_STYLE}. If set to this value, no expression evaluation will take place.
	 */
	final static String COMPONENT_DESCRIPTOR_STYLE_PLAIN = "plain";

	/**
	 * The class name of the component implementation. Many component types support the use of this property, which is why
	 * it is declared here. It is by no means a necessary requirement of a component type to require an implementation class.
	 * Component's that do support specification of a component implementation class do, by convention, use
	 * <a href="/javadoc/com.zfabrik.core.api%2Fjava/api/com/zfabrik/components/java/JavaComponentUtil.html#loadImplementationClassFromJavaComponent(java.lang.String,%20java.lang.String,%20com.zfabrik.resources.IResourceHandle)">JavaComponentUtil.html.loadImplementationClassFromJavaComponent()</a>
	 * to load that implementation class.
	 */
	final static String COMPONENT_CLZ  = "component.className";

	/**
	 * Components declare dependency on other components by specifying a comma-separated list of component names.
	 * All components will be queried for implementations of {@link IDependencyComponent} and correspondingly
	 * invoked before the component resource is even created. This is slightly different to dependencies for
	 * system states (see there) in that these dependencies will not be revisited upon synchronisations.
	 * The declaring component will be dependent on all declared dependency components.
	 */
	final static String DEPENDENCIES = "com.zfabrik.component.dependencies";

	/**
	 * Additional revision information. This may be some original information as provided by the source.
	 */
	final static String REVISION_INFO = "com.zfabrik.component.revisionInfo";

	/**
	 * A component may declare to an implementation of an extension point. This property is used to
	 * express a list of extension points implemented.
	 */
	final static String EXTENSION_POINTS = "com.zfabrik.component.extensions";

	/**
	 * Component type for component (symbolic) links. A component link delegates all lookups and resource retrievals to
	 * another, named component that is specified using the {@link #LINK_TARGET_COMPONENT} property. This is a low-level built-in
	 * component type.  
	 */
	public final static String LINK_COMPONENT_TYPE = "com.zfabrik.link";
	/**
	 * Component descriptor property defining the link target
	 */
	 final static String LINK_TARGET_COMPONENT = "link.targetComponent";
	
	/**
	 * Component type for "any" component. The any component type requires specification of {@link #COMPONENT_CLZ} and 
	 * has behavior depending on the type of class:
	 * <ul>
	 * <li>If the class implements {@link Resource}, all {@link Resource} calls will be delegated.</li>
	 * <li>If the class implements {@link Runnable}, it will be called upon system state attain operations.</li>
	 * <li>Otherwise a check will be performed if the class is castable to the requested type and if so, the a singleton instance will be
	 * returned</li>
	 * </ul>
	 * If no component type is specified on a component descriptor, it defaults to this type.
	 */
	 public static final String ANY_COMPONENT_TYPE = "com.zfabrik.any";
	
	/**
	 * Gets the name of the component.
	 */
	String getName();
	/**
	 * Gets the type of the component. A short-hand for <code>getProperty(COMPONENT_TYPE)</code>.
	 */
	String getType();

	/**
	 * Get the revision of the component. Minimum revision is zero. Revisions may not necessarily be strictly increasing,
	 * as components may be provided by different repositories over time.
	 */
	long   getRevision();

	/**
	 * Get the properties of this component. Note that these differ from the original configuration
	 * in these may be processed by some expression language evaluator.
	 */
	Properties getProperties();

	/**
	 * Get the raw properties of this component. Note that unlike {@link #getProperties()} the result of
	 * this method is data as defined in persistent component configuration-
	 */
	Properties getRawProperties();

	/**
	 * Just a short hand for <code>getProperties().getProperty(String)</code>
	 */
	String getProperty(String name);
}
