/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.components.provider.IComponentsRepositoryContext;
import com.zfabrik.impl.components.local.FSRepository;
import com.zfabrik.resources.IResourceManager;
import com.zfabrik.util.expression.X;
import com.zfabrik.work.CycleProtector;

public class ComponentsManager implements IComponentsManager {
	private boolean inited = false;
	private IComponentsRepository fsrepo;
	private CRListItem list = new CRListItem(this);

	// linked list that holds the repos according to their
	// prio (descending) and implements the repo context
	private static class CRListItem implements IComponentsRepositoryContext {
		CRListItem next,prev;
		int prio;
		IComponentsRepository repo;

		public CRListItem() {}

		// internal!
		private CRListItem(IComponentsRepository repo) {
			this.repo = repo;
			this.prio = Integer.MAX_VALUE;
		}

		public synchronized IComponentsRepository next() {
			if (this.next==null)
				return null;
			return this.next.repo;
		}

		public synchronized CRListItem insert(int prio, IComponentsRepository repo) {
			if ((this.next==null) || (this.next.prio<prio)) {
				// put it after this one
				CRListItem c = new CRListItem();
				c.repo = repo;
				c.prio = prio;
				c.next = this.next;
				c.prev = this;
				if (this.next != null)
					this.next.prev = c;
				this.next = c;
				return c;
			} else {
				return this.next.insert(prio, repo);
			}
		}

		public synchronized void remove(IComponentsRepository repo) {
			if (this.repo == repo) {
				if (this.prev!=null) 
					this.prev.next = this.next;
				if (this.next!=null)
					this.next.prev = this.prev;
			} else {
				this.next.remove(repo);
			}
		}

	}
	//

	public static void initialize() {
		((ComponentsManager)IComponentsManager.INSTANCE).init();
	}

	public static void shutdown() {
		((ComponentsManager)IComponentsManager.INSTANCE).destroy();
	}

	public synchronized void init() {
		if (inited) {
			throw new IllegalStateException("cannot initialize components manager twice!");
		}
		logger.fine("Initializing components manager");
		IResourceManager.INSTANCE.registerProvider(IComponentsLookup.COMPONENTS, new ComponentsProvider());

		// instantiate a root resource provider that will bootstrap
		// any other providers. It is a file system packages repository
		// that helps us use the same programming model to implement
		// remotable packages repositories.
		try {
			this.fsrepo = new FSRepository(this);
		} catch (IOException e) {
			throw new IllegalStateException("Failed to create and register FS repository", e);
		}
		this.inited = true;
	}

	public synchronized void destroy() {
		if (inited) {
			logger.fine("Shutting down the components manager");
			try {
				IResourceManager.INSTANCE.unregisterProvider(IComponentsLookup.COMPONENTS, true);
				this.unregisterRepository(this.fsrepo);
			} finally {
				this.inited = false;
			}
		}
	}

	public IComponentsRepositoryContext registerRepository(int prio, IComponentsRepository repo) {
		return this.list.insert(prio, repo);
	}

	public void unregisterRepository(IComponentsRepository repo) {
		this.list.remove(repo);
	}

	// unifying view over repos

	public Collection<String> findComponents(X propertyExpression) throws IOException {
		return findComponents(propertyExpression, false);
	}

	@Override
	public Collection<String> findComponents(X propertyExpression, boolean localOnly) throws IOException {
		if (localOnly) {
			return Collections.emptyList();
		}

		IComponentsRepository r = this.list.next();
		if (r!=null && ! localOnly) {
			return r.findComponents(propertyExpression);
		} 
		return Collections.emptyList();
	}
	
	public IComponentDescriptor getComponent(String component) {
		return getComponent(component, false);
	}

	@Override
	public IComponentDescriptor getComponent(String component, boolean localOnly) {
		if (localOnly) {
			return null;
		}
		
		IComponentsRepository r = this.list.next();
		if (r!=null && !localOnly) {
			return r.getComponent(component);
		} 
		return null;
	}

	public long getRevision(String component) throws IOException {
		return getRevision(component, false);
	}

	public long getRevision(String component, boolean localOnly) throws IOException {
		if (localOnly) {
			return -1;
		}
		
		IComponentsRepository r = this.list.next();
		if (r!=null) {
			return r.getRevision(component);
		} else {
			return -1;
		}
	}
	
	@Override
	public Set<String> getModules() throws IOException {
		return getModules(false);
	}

	@Override
	public Set<String> getModules(boolean localOnly) throws IOException {
		if (localOnly) {
			return Collections.emptySet();
		}
		IComponentsRepository r = this.list.next();
		if (r!=null) {
			return r.getModules(localOnly);
		} else {
			return Collections.emptySet();
		}
	}
	
	public File retrieve(String component) throws IOException {
		return retrieve(component, false);
	}

	@Override
	public File retrieve(String component, boolean localOnly) throws IOException {
		String node = "retrieve:"+component;
		CycleProtector.enter(node);
		try {
			if (localOnly) {
				return null;
			}
			
			// support links
			IComponentDescriptor d = getComponent(component);
			if (d!=null && IComponentDescriptor.LINK_COMPONENT_TYPE.equals(d.getType())) {
				String target = d.getProperty(IComponentDescriptor.LINK_TARGET_COMPONENT);
				if (target==null || (target=target.trim()).length()==0) {
					throw new IllegalStateException("Missing specification of "+IComponentDescriptor.LINK_TARGET_COMPONENT+": "+component);
				}
				return retrieve(target);
			}
			// end support links
			
			IComponentsRepository r = this.list.next();
			if (r!=null) {
				return r.retrieve(component);
			} 
			return null;
		} finally {
			CycleProtector.leave(node);
		}
	}

	private final static Logger logger = Logger.getLogger(ComponentsManager.class.getName());
}
